<?php
require_once __DIR__ . '/../config.php';

if (!function_exists('referral_log')) {
    function referral_log(string $level, string $event, array $context = []): void {
        $payload = [
            'level' => $level,
            'event' => $event,
            'context' => $context,
        ];
        error_log('[referral] ' . json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    }
}

if (!function_exists('referral_log_debug')) {
    function referral_log_debug(string $event, array $context = []): void {
        referral_log('debug', $event, $context);
    }
}

if (!function_exists('referral_log_info')) {
    function referral_log_info(string $event, array $context = []): void {
        referral_log('info', $event, $context);
    }
}

if (!function_exists('referral_log_error')) {
    function referral_log_error(string $event, array $context = []): void {
        referral_log('error', $event, $context);
    }
}

if (!function_exists('ensure_referral_schema')) {
    function ensure_referral_schema($connect): void {
        try { $connect->query("ALTER TABLE user ADD COLUMN referral_code VARCHAR(32) NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD UNIQUE KEY uq_user_referral_code (referral_code)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD COLUMN referred_by BIGINT NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD INDEX idx_user_referred_by (referred_by)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE user ADD COLUMN referral_opt_out TINYINT(1) NOT NULL DEFAULT 0"); } catch (Throwable $e) {}

        $connect->query("CREATE TABLE IF NOT EXISTS referral_rewards (
            id BIGINT PRIMARY KEY AUTO_INCREMENT,
            referrer_id BIGINT NOT NULL,
            invitee_id BIGINT NOT NULL,
            order_id BIGINT NOT NULL,
            code VARCHAR(64) NOT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY uniq_referrer_invitee (referrer_id, invitee_id),
            KEY idx_order (order_id),
            KEY idx_referrer (referrer_id),
            KEY idx_invitee (invitee_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");

        try { $connect->query("ALTER TABLE referral_rewards CHANGE COLUMN referred_user_id invitee_id BIGINT NOT NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards CHANGE COLUMN discount_code code VARCHAR(64) NOT NULL"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards DROP COLUMN discount_id"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards ADD COLUMN created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards DROP INDEX uq_referred_user"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards ADD UNIQUE KEY uniq_referrer_invitee (referrer_id, invitee_id)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards ADD KEY idx_order (order_id)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards ADD KEY idx_referrer (referrer_id)"); } catch (Throwable $e) {}
        try { $connect->query("ALTER TABLE referral_rewards ADD KEY idx_invitee (invitee_id)"); } catch (Throwable $e) {}

        try {
            $connect->query("INSERT IGNORE INTO messages (message_key, message_value) VALUES
                ('referral_share_message', '🎁 <b>برنامه دعوت دوستان</b>\\n\\nلینک اختصاصی شما:\\n<code>{referral_link}</code>\\n\\nکد دعوت: <code>{referral_code}</code>\\n\\nبا دعوت هر دوست و انجام اولین خرید او، یک کد تخفیف {reward_value} ({reward_type_label}) دریافت می‌کنید.\\n\\n👥 دعوت‌شدگان: {referral_total_invited}\\n🏆 پاداش‌های دریافت‌شده: {referral_success_count}\\n⏳ دعوت‌های در انتظار: {referral_pending_count}'),
                ('referral_reward_notification', '🎉 تبریک! دوست شما {referral_user_id} اولین خرید خود را انجام داد.\\n\\n🎟️ کد تخفیف هدیه شما:\\n<code>{discount_code}</code>\\n\\n📌 مقدار تخفیف: {reward_value} ({reward_type_label})\\n📦 حداقل مبلغ خرید: {min_order_amount}\\n⏰ تاریخ انقضا: {expires_at}\\n🏆 مجموع پاداش‌های شما: {reward_count}\\n\\nبرای استفاده کافی است هنگام خرید کد را وارد کنید.'),
                ('referral_disabled_message', '⚠️ برنامه دعوت دوستان در حال حاضر فعال نیست.'),
                ('referral_welcome_referred', '🎉 با دعوت دوست‌تان به ربات پیوستید! پس از اولین خرید شما، {referrer_id} یک هدیه ویژه دریافت می‌کند.')
            ");
        } catch (Throwable $e) {}
    }
}

if (!function_exists('referral_get_setting')) {
    function referral_get_setting($connect, string $key, $default = null) {
        static $cache = [];
        if (array_key_exists($key, $cache)) {
            return $cache[$key];
        }
        $stmt = $connect->prepare("SELECT value FROM settings WHERE `key` = ? LIMIT 1");
        $stmt->bind_param('s', $key);
        $stmt->execute();
        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();
        $cache[$key] = $row['value'] ?? $default;
        return $cache[$key];
    }
}

if (!function_exists('referral_get_settings')) {
    function referral_get_settings($connect): array {
        $keys = [
            'referral_status',
            'referral_reward_type',
            'referral_reward_value',
            'referral_min_order_total',
            'referral_code_ttl_days',
            'referral_reward_prefix',
            'referral_reward_limit',
            'referral_reward_min_order',
            'referral_reward_expires_days',
        ];
        $placeholders = implode(',', array_fill(0, count($keys), '?'));
        $stmt = $connect->prepare("SELECT `key`, `value` FROM settings WHERE `key` IN ($placeholders)");
        $stmt->bind_param(str_repeat('s', count($keys)), ...$keys);
        $stmt->execute();
        $res = $stmt->get_result();
        $settings = [];
        while ($row = $res->fetch_assoc()) {
            $settings[$row['key']] = $row['value'];
        }
        $stmt->close();

        $status = ($settings['referral_status'] ?? 'off') === 'on' ? 'on' : 'off';
        $reward_type = strtolower((string)($settings['referral_reward_type'] ?? 'fixed'));
        if (!in_array($reward_type, ['fixed', 'percent'], true)) {
            $reward_type = 'fixed';
        }
        $reward_value = (float)($settings['referral_reward_value'] ?? 0);
        if ($reward_type === 'percent') {
            $reward_value = max(1, min(100, $reward_value));
        } else {
            $reward_value = max(0, $reward_value);
        }
        $raw_min = $settings['referral_min_order_total'] ?? $settings['referral_reward_min_order'] ?? 0;
        $raw_ttl = $settings['referral_code_ttl_days'] ?? $settings['referral_reward_expires_days'] ?? 0;
        $min_order = max(0, (float)$raw_min);
        $ttl_days = max(0, (int)$raw_ttl);
        $prefix = strtoupper(preg_replace('/[^A-Z0-9]/i', '', (string)($settings['referral_reward_prefix'] ?? 'REF')));
        $prefix = substr($prefix ?: 'REF', 0, 12);
        $limit = max(1, (int)($settings['referral_reward_limit'] ?? 1));

        return [
            'status' => $status,
            'reward_type' => $reward_type,
            'reward_value' => $reward_value,
            'reward_min_order' => $min_order,
            'reward_expires_days' => $ttl_days,
            'reward_prefix' => $prefix,
            'reward_limit' => $limit,
        ];
    }
}

if (!function_exists('referral_is_enabled')) {
    function referral_is_enabled($connect): bool {
        static $cached = null;
        if ($cached === null) {
            $cached = referral_get_settings($connect);
        }
        return ($cached['status'] ?? 'off') === 'on';
    }
}

if (!function_exists('referral_generate_random_code')) {
    function referral_generate_random_code(int $length = 8): string {
        $length = max(4, min(32, $length));
        try {
            $raw = bin2hex(random_bytes($length));
        } catch (Throwable $e) {
            $raw = md5(uniqid((string)mt_rand(), true));
        }
        return strtoupper(substr($raw, 0, $length));
    }
}

if (!function_exists('generateUniqueDiscountCode')) {
    function generateUniqueDiscountCode($connect, ?string $prefix = null): string {
        if ($prefix === null) {
            $settings = referral_get_settings($connect);
            $prefix = $settings['reward_prefix'] ?? 'REF';
        }
        $prefix = strtoupper(preg_replace('/[^A-Z0-9]/i', '', (string)$prefix));
        $prefix = $prefix ? $prefix . '-' : '';
        do {
            $code = $prefix . referral_generate_random_code(6);
            $stmt = $connect->prepare("SELECT id FROM discount_codes WHERE code = ? LIMIT 1");
            $stmt->bind_param('s', $code);
            $stmt->execute();
            $exists = $stmt->get_result()->fetch_assoc();
            $stmt->close();
        } while ($exists);
        return $code;
    }
}

if (!function_exists('referral_generate_unique_discount_code')) {
    function referral_generate_unique_discount_code($connect, string $prefix): string {
        return generateUniqueDiscountCode($connect, $prefix);
    }
}

if (!function_exists('referral_generate_unique_user_code')) {
    function referral_generate_unique_user_code($connect): string {
        do {
            $code = referral_generate_random_code(8);
            $stmt = $connect->prepare("SELECT id FROM user WHERE referral_code = ? LIMIT 1");
            $stmt->bind_param('s', $code);
            $stmt->execute();
            $exists = $stmt->get_result()->fetch_assoc();
            $stmt->close();
        } while ($exists);
        return $code;
    }
}

if (!function_exists('referral_ensure_user_code')) {
    function referral_ensure_user_code($connect, string $user_id): string {
        $stmt = $connect->prepare("SELECT referral_code FROM user WHERE id = ? LIMIT 1");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();
        if (!empty($row['referral_code'])) {
            return $row['referral_code'];
        }
        $code = referral_generate_unique_user_code($connect);
        $stmt = $connect->prepare("UPDATE user SET referral_code = ? WHERE id = ? AND (referral_code IS NULL OR referral_code = '' )");
        $stmt->bind_param('ss', $code, $user_id);
        $stmt->execute();
        $stmt->close();
        return $code;
    }
}

if (!function_exists('referral_user_has_referrer')) {
    function referral_user_has_referrer($connect, int $user_id): bool {
        $stmt = $connect->prepare("SELECT referred_by FROM user WHERE id = ? LIMIT 1");
        $stmt->bind_param('i', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();
        $ref = $row['referred_by'] ?? null;
        return !empty($ref);
    }
}

if (!function_exists('referral_resolve_referrer')) {
    function referral_resolve_referrer($connect, ?string $payload, int $inviteeUserId): ?int {
        if (!$payload) {
            referral_log_debug('referral_start_payload_empty', ['invitee_id' => $inviteeUserId]);
            return null;
        }
        if ((getSetting($connect, 'referral_status', 'off') ?? 'off') !== 'on') {
            referral_log_debug('referral_program_disabled', ['invitee_id' => $inviteeUserId]);
            return null;
        }
        $code = strtoupper(trim($payload));
        $code = preg_replace('/[^A-Z0-9_-]/i', '', $code);
        if ($code === '') {
            referral_log_debug('referral_payload_invalid', ['invitee_id' => $inviteeUserId]);
            return null;
        }
        $stmt = $connect->prepare("SELECT id FROM user WHERE referral_code = ? LIMIT 1");
        $stmt->bind_param('s', $code);
        $stmt->execute();
        $res = $stmt->get_result();
        $row = $res->fetch_assoc();
        $stmt->close();
        if (!$row) {
            referral_log_debug('referral_referrer_not_found', ['invitee_id' => $inviteeUserId, 'code' => $code]);
            return null;
        }
        $referrerId = (int)$row['id'];
        if ($referrerId === $inviteeUserId) {
            referral_log_debug('referral_self_reference_blocked', ['user_id' => $inviteeUserId]);
            return null;
        }
        return $referrerId;
    }
}

if (!function_exists('referral_assign_referred_by')) {
    function referral_assign_referred_by($connect, int $user_id, int $referrer_id): bool {
        if ($user_id === $referrer_id) {
            return false;
        }
        $stmt = $connect->prepare("UPDATE user SET referred_by = ? WHERE id = ? AND (referred_by IS NULL OR referred_by = '' OR referred_by = '0')");
        $stmt->bind_param('ii', $referrer_id, $user_id);
        $stmt->execute();
        $affected = $connect->affected_rows > 0;
        $stmt->close();
        return $affected;
    }
}

if (!function_exists('referral_format_currency')) {
    function referral_format_currency($amount): string {
        return number_format((int)$amount) . ' تومان';
    }
}

if (!function_exists('referral_format_datetime')) {
    function referral_format_datetime(?string $datetime): string {
        if (empty($datetime)) {
            return 'بدون محدودیت';
        }
        try {
            $dt = new DateTime($datetime, new DateTimeZone('Asia/Tehran'));
        } catch (Throwable $e) {
            try { $dt = new DateTime($datetime); } catch (Throwable $ex) { return $datetime; }
        }
        if (function_exists('jdate')) {
            return jdate('Y/m/d H:i', $dt->getTimestamp());
        }
        return $dt->format('Y-m-d H:i');
    }
}

if (!function_exists('referral_get_user_stats')) {
    function referral_get_user_stats($connect, string $user_id): array {
        $stats = [
            'invited' => 0,
            'successful' => 0,
            'pending' => 0,
        ];
        $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM user WHERE referred_by = ?");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $stats['invited'] = (int)($res->fetch_assoc()['cnt'] ?? 0);
        $stmt->close();

        $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM referral_rewards WHERE referrer_id = ?");
        $stmt->bind_param('s', $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        $stats['successful'] = (int)($res->fetch_assoc()['cnt'] ?? 0);
        $stmt->close();

        $stats['pending'] = max(0, $stats['invited'] - $stats['successful']);
        return $stats;
    }
}

if (!function_exists('referral_get_bot_username')) {
    function referral_get_bot_username($connect): ?string {
        $username = referral_get_setting($connect, 'bot_username', '');
        if (!empty($username)) {
            return $username;
        }
        if (!function_exists('bot')) {
            return null;
        }
        $response = bot('getMe');
        $username = $response['result']['username'] ?? '';
        if (!empty($username)) {
            $username = trim($username);
            $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('bot_username', ?) ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)");
            $stmt->bind_param('s', $username);
            $stmt->execute();
            $stmt->close();
            return $username;
        }
        return null;
    }
}

if (!function_exists('referral_build_link')) {
    function referral_build_link($connect, string $code): ?string {
        $username = referral_get_bot_username($connect);
        if (!$username) {
            return null;
        }
        return 'https://t.me/' . $username . '?start=' . urlencode($code);
    }
}

if (!function_exists('getSetting')) {
    function getSetting($connect, string $key, $default = null) {
        return referral_get_setting($connect, $key, $default);
    }
}

if (!function_exists('countApprovedOrders')) {
    function countApprovedOrders($connect, int $userId): int {
        $stmt = $connect->prepare("SELECT COUNT(*) AS cnt FROM orders WHERE user_id = ? AND status='approved'");
        $stmt->bind_param('i', $userId);
        $stmt->execute();
        $res = $stmt->get_result();
        $count = (int)($res->fetch_assoc()['cnt'] ?? 0);
        $stmt->close();
        return $count;
    }
}

if (!function_exists('referralRewardExists')) {
    function referralRewardExists($connect, int $referrerId, int $inviteeId): bool {
        $stmt = $connect->prepare("SELECT id FROM referral_rewards WHERE referrer_id = ? AND invitee_id = ? LIMIT 1");
        $stmt->bind_param('ii', $referrerId, $inviteeId);
        $stmt->execute();
        $res = $stmt->get_result();
        $exists = $res->fetch_assoc() ? true : false;
        $stmt->close();
        return $exists;
    }
}

if (!function_exists('buildReferralReward')) {
    function buildReferralReward($connect, array $config): ?array {
        $type = strtolower((string)($config['reward_type'] ?? 'fixed'));
        if (!in_array($type, ['fixed', 'percent'], true)) {
            $type = 'fixed';
        }
        $value = (float)($config['reward_value'] ?? 0);
        if ($value <= 0) {
            return null;
        }
        if ($type === 'percent') {
            $value = max(1, min(100, $value));
        } else {
            $value = max(0, $value);
        }
        $minOrder = max(0, (float)($config['min_order'] ?? 0));
        $ttlDays = max(0, (int)($config['ttl_days'] ?? 0));
        $expiresAt = $ttlDays > 0 ? nowPlusDays($ttlDays) : null;
        return [
            'referrer_id' => (int)($config['referrer_id'] ?? 0),
            'invitee_id' => (int)($config['invitee_id'] ?? 0),
            'base_order_id' => (int)($config['base_order_id'] ?? 0),
            'reward_type' => $type,
            'reward_value' => $value,
            'min_order' => $minOrder,
            'ttl_days' => $ttlDays,
            'expires_at' => $expiresAt,
        ];
    }
}

if (!function_exists('buildReferralRewardMessage')) {
    function buildReferralRewardMessage(string $code, array $reward): string {
        $typeLabel = $reward['reward_type'] === 'percent' ? 'درصدی' : 'تومانی';
        if ($reward['reward_type'] === 'percent') {
            $valueText = rtrim(rtrim(sprintf('%.2f', $reward['reward_value']), '0'), '.') . '%';
        } else {
            $valueText = referral_format_currency($reward['reward_value']);
        }
        $minOrderText = $reward['min_order'] > 0 ? referral_format_currency($reward['min_order']) : 'ندارد';
        $expiresText = !empty($reward['expires_at']) ? referral_format_datetime($reward['expires_at']) : 'بدون محدودیت';
        return "تبریک! به خاطر خرید دوست شما، یک کد تخفیف برایتان ساخته شد:\n<code>{$code}</code>\nنوع: {$typeLabel} مقدار: {$valueText} حداقل خرید: {$minOrderText}\nاعتبار تا: {$expiresText}";
    }
}

if (!function_exists('getTelegramChatIdByUserId')) {
    function getTelegramChatIdByUserId(int $userId): ?int {
        return $userId > 0 ? $userId : null;
    }
}

if (!function_exists('buildShareKeyboard')) {
    function buildShareKeyboard(string $code): ?array {
        if ($code === '') {
            return null;
        }
        return [
            'inline_keyboard' => [
                [
                    ['text' => 'اشتراک‌گذاری کد', 'switch_inline_query' => $code]
                ]
            ]
        ];
    }
}

if (!function_exists('nowPlusDays')) {
    function nowPlusDays(int $days): string {
        try {
            $dt = new DateTime('now', new DateTimeZone('Asia/Tehran'));
        } catch (Throwable $e) {
            $dt = new DateTime();
        }
        $dt->modify('+' . $days . ' days');
        return $dt->format('Y-m-d H:i:s');
    }
}

if (!function_exists('insertDiscountCode')) {
    function insertDiscountCode($connect, array $data): int {
        $sql = "INSERT INTO discount_codes (code, owner_user_id, type, value, max_uses, per_user_limit, min_order_total, min_order_amount, expires_at, is_active, note, meta) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 1, ?, ?)";
        $stmt = $connect->prepare($sql);
        $code = (string)$data['code'];
        $owner = (int)$data['owner_user_id'];
        $type = (string)$data['type'];
        $value = (float)$data['value'];
        $maxUses = (int)($data['max_uses'] ?? 1);
        $perUser = (int)($data['per_user_limit'] ?? 1);
        $minOrderTotal = (float)($data['min_order_total'] ?? 0);
        $minOrderAmount = (float)($data['min_order_amount'] ?? $minOrderTotal);
        $expiresAt = $data['expires_at'] ?? null;
        $note = $data['note'] ?? null;
        $meta = $data['meta'] ?? null;
        $stmt->bind_param('sisdiiddsss', $code, $owner, $type, $value, $maxUses, $perUser, $minOrderTotal, $minOrderAmount, $expiresAt, $note, $meta);
        $stmt->execute();
        $id = $stmt->insert_id;
        $stmt->close();
        return (int)$id;
    }
}

if (!function_exists('insertReferralReward')) {
    function insertReferralReward($connect, array $data): bool {
        $sql = "INSERT IGNORE INTO referral_rewards (referrer_id, invitee_id, order_id, code) VALUES (?, ?, ?, ?)";
        $stmt = $connect->prepare($sql);
        $referrer = (int)$data['referrer_id'];
        $invitee = (int)$data['invitee_id'];
        $orderId = (int)$data['order_id'];
        $code = (string)$data['code'];
        $stmt->bind_param('iiis', $referrer, $invitee, $orderId, $code);
        $stmt->execute();
        $affected = $connect->affected_rows > 0;
        $stmt->close();
        return $affected;
    }
}

if (!function_exists('tg_sendMessage')) {
    function tg_sendMessage($chatId, string $text, array $options = []) {
        if (isset($GLOBALS['__referral_test_sender']) && is_callable($GLOBALS['__referral_test_sender'])) {
            return call_user_func($GLOBALS['__referral_test_sender'], $chatId, $text, $options);
        }
        $params = array_merge([
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => 'html',
            'disable_web_page_preview' => true,
        ], $options);
        if (isset($params['reply_markup']) && is_array($params['reply_markup'])) {
            $params['reply_markup'] = json_encode($params['reply_markup'], JSON_UNESCAPED_UNICODE);
        }
        if (function_exists('bot')) {
            return bot('sendMessage', $params);
        }
        if (function_exists('sendmessage')) {
            $replyMarkup = $params['reply_markup'] ?? null;
            return sendmessage($chatId, $text, $replyMarkup, 'html');
        }
        if (function_exists('sendMessage')) {
            $replyMarkup = $params['reply_markup'] ?? null;
            return sendMessage($chatId, $text, $replyMarkup, 'html');
        }
        return null;
    }
}

if (!function_exists('referral_process_successful_order')) {
    function referral_process_successful_order($connect, array $order): void {
        $status = strtolower((string)($order['status'] ?? ''));
        if ((getSetting($connect, 'referral_status', 'off') ?? 'off') !== 'on') {
            referral_log_debug('referral_program_disabled_on_order', ['order_id' => $order['id'] ?? null]);
            return;
        }
        if ($status !== 'approved') {
            referral_log_debug('referral_order_not_approved', ['order_id' => $order['id'] ?? null, 'status' => $status]);
            return;
        }
        if (($order['type'] ?? '') === 'exten') {
            referral_log_debug('referral_skip_renewal', ['order_id' => $order['id'] ?? null]);
            return;
        }
        $orderId = (int)($order['id'] ?? 0);
        $inviteeId = (int)($order['user_id'] ?? 0);
        if ($orderId <= 0 || $inviteeId <= 0) {
            referral_log_debug('referral_missing_order_data', ['order' => $order]);
            return;
        }

        $stmt = $connect->prepare("SELECT id, referred_by FROM user WHERE id = ? LIMIT 1");
        $stmt->bind_param('i', $inviteeId);
        $stmt->execute();
        $res = $stmt->get_result();
        $userRow = $res->fetch_assoc();
        $stmt->close();
        if (!$userRow) {
            referral_log_debug('referral_invitee_not_found', ['user_id' => $inviteeId]);
            return;
        }
        $referrerId = (int)($userRow['referred_by'] ?? 0);
        if ($referrerId <= 0 || $referrerId === $inviteeId) {
            referral_log_debug('referral_no_referrer', ['user_id' => $inviteeId, 'referrer_id' => $referrerId]);
            return;
        }

        if (countApprovedOrders($connect, $inviteeId) !== 1) {
            referral_log_debug('referral_not_first_order', ['user_id' => $inviteeId]);
            return;
        }

        if (referralRewardExists($connect, $referrerId, $inviteeId)) {
            referral_log_debug('referral_reward_already_exists', ['referrer_id' => $referrerId, 'invitee_id' => $inviteeId]);
            return;
        }

        $settings = referral_get_settings($connect);
        $reward = buildReferralReward($connect, [
            'referrer_id' => $referrerId,
            'invitee_id' => $inviteeId,
            'base_order_id' => $orderId,
            'reward_type' => $settings['reward_type'] ?? getSetting($connect, 'referral_reward_type', 'fixed'),
            'reward_value' => $settings['reward_value'] ?? (float)getSetting($connect, 'referral_reward_value', 0),
            'min_order' => $settings['reward_min_order'] ?? (float)getSetting($connect, 'referral_min_order_total', 0),
            'ttl_days' => $settings['reward_expires_days'] ?? (int)getSetting($connect, 'referral_code_ttl_days', 0),
        ]);
        if (!$reward) {
            referral_log_debug('referral_no_reward_configured', ['order_id' => $orderId]);
            return;
        }

        $connect->begin_transaction();
        try {
            $code = generateUniqueDiscountCode($connect, $settings['reward_prefix'] ?? null);
            $note = sprintf('Referral reward for invitee %d (order %d)', $inviteeId, $orderId);
            $meta = json_encode([
                'kind' => 'referral_reward',
                'referrer_id' => $referrerId,
                'invitee_id' => $inviteeId,
                'base_order_id' => $orderId,
            ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

            insertDiscountCode($connect, [
                'code' => $code,
                'owner_user_id' => $referrerId,
                'type' => $reward['reward_type'],
                'value' => $reward['reward_value'],
                'max_uses' => 1,
                'per_user_limit' => 1,
                'min_order_total' => $reward['min_order'],
                'min_order_amount' => $reward['min_order'],
                'expires_at' => $reward['expires_at'],
                'note' => $note,
                'meta' => $meta,
            ]);

            if (!insertReferralReward($connect, [
                'referrer_id' => $referrerId,
                'invitee_id' => $inviteeId,
                'order_id' => $orderId,
                'code' => $code,
            ])) {
                $connect->rollback();
                referral_log_debug('referral_duplicate_guard_triggered', ['referrer_id' => $referrerId, 'invitee_id' => $inviteeId]);
                return;
            }

            $connect->commit();
        } catch (Throwable $e) {
            $connect->rollback();
            referral_log_error('referral_reward_fail', ['order_id' => $orderId, 'error' => $e->getMessage()]);
            return;
        }

        $chatId = getTelegramChatIdByUserId($referrerId);
        if ($chatId) {
            $message = buildReferralRewardMessage($code, $reward);
            $keyboard = buildShareKeyboard($code);
            $options = [];
            if ($keyboard) {
                $options['reply_markup'] = $keyboard;
            }
            tg_sendMessage($chatId, $message, $options);
        }

        referral_log_info('referral_reward_issued', [
            'order_id' => $orderId,
            'referrer_id' => $referrerId,
            'invitee_id' => $inviteeId,
        ]);
    }
}
