<?php
// =========================================================================================================
// --- INCLUDES & CONFIG ---
// =========================================================================================================
include_once('class.php');
require_once __DIR__ . '/lib/discounts.php';
require_once __DIR__ . '/lib/server_domains.php';
require_once __DIR__ . '/lib/server_metadata.php';
require_once __DIR__ . '/lib/image_helper.php';
require_once __DIR__ . '/lib/referrals.php';
// Ensure discount-related tables/columns are present
ensure_discount_schema($connect);
// Ensure servers table has custom_domain column
ensure_server_domain_column($connect);
ensure_server_metadata_schema($connect);
ensure_referral_schema($connect);
// Ensure columns for star-based pricing exist
try { $connect->query("ALTER TABLE plan_prices ADD COLUMN stars_price INT NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
try { $connect->query("ALTER TABLE orders ADD COLUMN stars_amount INT NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
// Ensure default payment messages exist
try {
    $connect->query("INSERT IGNORE INTO messages (message_key, message_value) VALUES
        ('crypto_payment_message', '💎 <b>پرداخت با رمز ارز</b>\\n\\nلطفاً مبلغ معادل <b>{amount_toman} تومان</b>{amount_usdt} را به یکی از کیف پول‌های زیر واریز نمایید:\\n\\n<code>\\n{wallets}\\n</code>\\n\\n‼️ <b>مهم:</b> پس از واریز، لطفاً <b>اسکرین‌شات از رسید پرداخت</b> را به همراه <b>لینک یا هش تراکنش (TxID)</b> در <b>کپشن همان اسکرین‌شات</b> برای ما ارسال کنید.\\n\\nسفارش شما پس از تایید تراکنش توسط ادمین، فعال خواهد شد.'),
        ('stars_payment_message', 'برای پرداخت با استارز از فاکتور ارسال‌شده استفاده کنید.')");
} catch (Throwable $e) {}
date_default_timezone_set('Asia/Tehran');

// Telegram Stars pricing: 100 Stars = $1.69
if (!defined('STAR_PRICE_USD_PER_100')) {
    define('STAR_PRICE_USD_PER_100', 1.69);
}

// =========================================================================================================
// --- LOCAL HELPERS FOR PAYMENT MESSAGE/IMAGE ---
// =========================================================================================================

/**
 * Build safe HTML for the card-payment instruction, fixing font issues by:
 * - Converting backticked parts to <code> only if they are Latin/digits
 * - Removing <code> around Arabic/Persian text to avoid broken monospace shaping
 */
function normalize_payment_html($html) {
    // Convert backticks to <code> for Latin-only segments
    $html = preg_replace_callback('/`([^`]+)`/u', function($m) {
        $inner = $m[1];
        if (preg_match('/\p{Arabic}/u', $inner)) return $inner; // keep plain if Arabic
        return '<code>' . htmlspecialchars($inner, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '</code>';
    }, (string)$html);
    // Strip <code> wrappers that contain Arabic to avoid monospace issues
    $html = preg_replace_callback('/<code>(.*?)<\/code>/us', function($m){
        $inner = html_entity_decode($m[1], ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
        if (preg_match('/\p{Arabic}/u', $inner)) return $inner; // remove code tag
        return '<code>' . htmlspecialchars($inner, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8') . '</code>';
    }, $html);
    return $html;
}

/**
 * Send payment instruction either as a rendered photo (using image_helper)
 * or as a plain HTML message, based on settings.
 */
function send_card_payment_instruction($connect, $chat_id, $order_code, $final_amount_text, $back_key, $prefix_html = '') {
    // Build the message HTML from templates/settings
    $card_info = get_setting($connect, 'card_info');
    $raw = getMessage($connect, 'card_payment_instruction', [
        'order_code' => $order_code,
        'amount'     => $final_amount_text,
        'card_info'  => $card_info,
    ]);
    $html = normalize_payment_html($raw);
    if ($prefix_html) { $html = trim($prefix_html) . "\n\n" . $html; }

    // Decide whether to send as image
    $img_enabled = get_setting($connect, 'payment_image_enabled');
    if ($img_enabled === null || $img_enabled === '') { $img_enabled = 'on'; }

    if ($img_enabled === 'on') {
        // Choose template image to send as-is (no text drawing)
        $template_path = null;
        $tpl_setting = get_setting($connect, 'payment_template_path');
        if (!empty($tpl_setting)) {
            $maybe = __DIR__ . '/' . ltrim($tpl_setting, '/');
            if (is_readable($maybe)) $template_path = $maybe;
        }
        if (!$template_path) {
            $maybe1 = __DIR__ . '/assets/payment_template.jpg';
            $maybe2 = __DIR__ . '/assets/payment_template.png';
            if (is_readable($maybe1)) $template_path = $maybe1; elseif (is_readable($maybe2)) $template_path = $maybe2;
        }
        if (!$template_path && function_exists('ih_default_template')) {
            $template_path = ih_default_template();
        }
        if ($template_path && is_readable($template_path)) {
            return bot('sendPhoto', [
                'chat_id' => $chat_id,
                'photo'   => new CURLFile(realpath($template_path)),
                'caption' => $html,
                'parse_mode' => 'html',
                'reply_markup' => $back_key,
            ]);
        }
    }

    return sendMessage($chat_id, $html, $back_key, 'html');
}

// =========================================================================================================
// --- DATABASE-DRIVEN HELPER FUNCTIONS ---
// =========================================================================================================

function get_usdt_price() {
    $curl = curl_init();
    curl_setopt_array($curl, array(
        CURLOPT_URL => 'https://api.bitpin.ir/v1/mkt/markets/',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => '',
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => 'GET',
    ));
    if (defined('DISABLE_SSL_VERIFY') && DISABLE_SSL_VERIFY) {
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($curl, CURLOPT_SSL_VERIFYHOST, 0);
    }
    $response = curl_exec($curl);
    curl_close($curl);
    $data = json_decode($response, true);

    if (isset($data['results'])) {
        foreach ($data['results'] as $market) {
            if ($market['code'] === 'USDT_IRT') {
                return (float)$market['price'];
            }
        }
    }
    return null;
}

$message_cache = [];
function getMessage($connect, $key, $placeholders = []) {
    global $message_cache;
    if (!isset($message_cache[$key])) {
        $stmt = $connect->prepare("SELECT message_value FROM messages WHERE message_key = ? LIMIT 1");
        $stmt->bind_param("s", $key);
        $stmt->execute();
        $result = $stmt->get_result();
        $message_cache[$key] = ($row = $result->fetch_assoc()) ? $row['message_value'] : "خطای پیام: کلید '{$key}' یافت نشد.";
        $stmt->close();
    }
 
    $message = $message_cache[$key];
    foreach ($placeholders as $placeholder => $value) {
        $message = str_replace('{' . $placeholder . '}', $value, $message);
    }
    return $message;
}

$settings_cache = [];
function get_setting($connect, $key) {
    global $settings_cache;
    if (!isset($settings_cache[$key])) {
        $stmt = $connect->prepare("SELECT value FROM settings WHERE `key` = ? LIMIT 1");
        $stmt->bind_param("s", $key);
        $stmt->execute();
        $result = $stmt->get_result();
        $settings_cache[$key] = ($row = $result->fetch_assoc()) ? $row['value'] : null;
        $stmt->close();
    }
    return $settings_cache[$key];
}

function joinChannels($connect, $chat_id) {
    $channels_str = get_setting($connect, 'join_channels');
    if (empty($channels_str)) return true;
    $channels = array_filter(array_map('trim', explode("\n", $channels_str)));
    $not_joined = [];
    foreach ($channels as $channel) {
        if (empty($channel)) continue;
        $resp = bot('getChatMember', ['chat_id' => $channel, 'user_id' => $chat_id]);
        $tch = $resp['result']['status'] ?? 'left';
        if (!in_array($tch, ['creator', 'administrator', 'member'])) {
            $not_joined[] = $channel;
        }
    }
    return empty($not_joined) ? true : $not_joined;
}

// =========================================================================================================
// --- UI HELPER FUNCTIONS ---
// =========================================================================================================

/**
 * Creates a text-based progress bar for Telegram messages.
 * @param int $percentage The percentage value (0-100).
 * @param int $length The total character length of the bar.
 * @return string The generated progress bar.
 */
function create_progress_bar($percentage, $length = 10) {
    $filled_length = round($length * $percentage / 100);
    $empty_length = $length - $filled_length;
    return str_repeat('▓', $filled_length) . str_repeat('░', $empty_length);
}

function format_number_plain($value) {
    return number_format((int)$value);
}

function format_decimal_plain($value, $precision = 1) {
    $precision = max(0, (int)$precision);
    $formatted = number_format((float)$value, $precision, '.', '');
    if ($precision > 0) {
        $formatted = rtrim(rtrim($formatted, '0'), '.');
    }
    return $formatted;
}

function format_currency_toman($amount) {
    return number_format((int)$amount) . ' تومان';
}

function payment_type_label($type) {
    $type = strtolower((string)$type);
    switch ($type) {
        case 'card':
        case 'card2card':
            return 'کارت به کارت';
        case 'crypto':
        case 'usdt':
            return 'رمزارز';
        case 'stars':
        case 'telegram':
            return 'استارز تلگرام';
        case 'free':
            return 'رایگان';
        default:
            return ($type === 'unknown' ? 'نامشخص' : $type);
    }
}

function escape_html($value) {
    return htmlspecialchars((string)$value, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

// Ensure a user row exists so steps can persist for first-time users
function ensure_user_exists($connect, $uid) {
    if (empty($uid)) return;
    $sql = "INSERT INTO user (id, step) SELECT ?, 'none' FROM DUAL WHERE NOT EXISTS (SELECT 1 FROM user WHERE id = ?)";
    $stmt = $connect->prepare($sql);
    $stmt->bind_param('ss', $uid, $uid);
    $stmt->execute();
    $stmt->close();
}

/**
 * Edit message in-place for both text and media messages.
 * If the callback originated from a media message (photo/video/document/...)
 * Telegram requires editing the caption, not the text. This helper detects
 * the message type from the callback payload and calls the right API.
 */
function editMessageSmart($chat_id, $message_id, $text, $keyboard = null, $pars = 'html') {
    // $update2 is populated in class.php and available here after include
    global $update2;
    $is_media = false;
    if (isset($update2['callback_query']['message'])) {
        $msg = $update2['callback_query']['message'];
        if (isset($msg['photo']) || isset($msg['video']) || isset($msg['document']) || isset($msg['animation']) || isset($msg['audio'])) {
            $is_media = true;
        }
    }
    if ($is_media) {
        return editMessageCaption($chat_id, $message_id, $text, $keyboard, $pars);
    }
    return editMessageText($chat_id, $message_id, $text, $keyboard, $pars);
}

// =========================================================================================================
// --- CORE BOT FUNCTIONS ---
// =========================================================================================================
function handlePaymentConfirmation($connect, $action, $order_code, $chatid, $messageid) {
    global $start; 
    
    $stmt = $connect->prepare("SELECT * FROM orders WHERE order_code = ? LIMIT 1");
    $stmt->bind_param("s", $order_code);
    $stmt->execute();
    $order_res = $stmt->get_result();

    if ($order_res->num_rows == 0) {
        if (!empty($GLOBALS['membercall'])) {
            bot('answercallbackquery', ['callback_query_id' => $GLOBALS['membercall'], 'text' => '⛔️ سفارش یافت نشد!', 'show_alert' => true]);
        }
        $stmt->close();
        return;
    }
    $order = $order_res->fetch_assoc();
    $stmt->close();

    $user_id = $order['user_id'];
    $size = $order['size'];
    $amount = number_format($order['amount']);
    $loc = $order['location'];
    $days = $order['days'];
    $order_type = $order['type'];
    $svcode = $order['svcode'];

    $stmt_server = $connect->prepare("SELECT * FROM servers WHERE name = ? AND status = 'on' LIMIT 1");
    $stmt_server->bind_param("s", $loc);
    $stmt_server->execute();
    $server_res = $stmt_server->get_result();

    if ($server_res->num_rows == 0) {
        $alert_text = getMessage($connect, 'error_server_not_found_or_off');
        if (!empty($GLOBALS['membercall'])) {
            bot('answercallbackquery', ['callback_query_id' => $GLOBALS['membercall'], 'text' => $alert_text, 'show_alert' => true]);
        }
        $stmt_server->close();
        return;
    }
    $server = $server_res->fetch_assoc();
    $server_display_name = $server['display_name'];
    $stmt_server->close();

    if ($action == 'approve') {
        $type = $server['type'];
        $link = $server['link'];
        $username = $server['username'] ?: 'ندارد';
        $password = $server['password'] ?: 'ندارد';
        $inbound_copy = $server['inbound_copy'] ?: 'ندارد';
        $list = $server['protocols'];

        $stmt_update_order = $connect->prepare("UPDATE orders SET status='approved' WHERE order_code=?");
        $stmt_update_order->bind_param("s", $order_code);
        $stmt_update_order->execute();
        $stmt_update_order->close();
        // Confirm discount usage (if applied)
        if (function_exists('confirm_discount_usage_for_order')) {
            confirm_discount_usage_for_order($connect, $order_code);
        }

        $alert_text = getMessage($connect, 'admin_approve_payment_alert');
        if (!empty($GLOBALS['membercall'])) {
            bot('answercallbackquery', ['callback_query_id' => $GLOBALS['membercall'], 'text' => $alert_text, 'show_alert' => true]);
        }
        if ($messageid) {
            editMessageCaption($chatid, $messageid, "🧾 فیش واریزی تأیید شد:\n\n👤 کاربر: $user_id\n📦 حجم: $size گیگ\n💸 مبلغ: $amount تومان\n📍 لوکیشن: $server_display_name\n\n✅ وضعیت: تأیید شده");
        }

        if ($order_type == "exten") {
            $stmt_service = $connect->prepare("SELECT * FROM services WHERE id = ? LIMIT 1");
            $stmt_service->bind_param("s", $svcode);
            $stmt_service->execute();
            $service_res = $stmt_service->get_result();
            if($service_res->num_rows > 0){
                $service = $service_res->fetch_assoc();
                $name = $service['name'];
                $current_expire_time = (int)($service['expire'] ?? 0);
                $base_time = $current_expire_time > time() ? $current_expire_time : time();
                $days_to_extend = max(0, (int)$days);
                $new_expire_time = $days_to_extend > 0 ? strtotime("+{$days_to_extend} day", $base_time) : $base_time;

                include_once('jdf.php');
                $Y_new = date('Y', $new_expire_time);
                $m_new = date('m', $new_expire_time);
                $d_new = date('d', $new_expire_time);
                $new_expire_date_jalali = gregorian_to_jalali($Y_new, $m_new, $d_new, '/');

                if ($type == "marzban") {
                    $token = loginPanel($link, $username, $password)['access_token'];
                    $proxies = buildProxies($list, 'on');
                    resetUser($name, $token, $link);
                    tamdid($name, convertToBytes($size . 'GB'), $new_expire_time, $proxies, $token, $link);
                } elseif ($type == "sanaei") {
                    include_once("api/sanaei.php");
                    login($link, $username, $password);
                    $uuid = $service['uuid'];
                    ResetClientTraffic($name, $inbound_copy, $link);
                    UpdateClient($uuid, $name, $size, $new_expire_time * 1000, "$link", "$inbound_copy");
                }
                
                $stmt_update_service = $connect->prepare("UPDATE `services` SET size=?, expire=?, volume_notified='0' WHERE id=?");
                $stmt_update_service->bind_param("iis", $size, $new_expire_time, $svcode);
                $stmt_update_service->execute();
                $stmt_update_service->close();

                $key = json_encode(['inline_keyboard' => [[['text' => "🎛 مدیریت سرویس", 'callback_data' => "service-$svcode"]]]]);
                $sub = apply_custom_domain($service['sub'], $server);
                $encode_url = urlencode($sub);
                $caption = "✅ <b>سرویس شما با موفقیت تمدید شد</b>\n\n" . " اشتراک شما با اطلاعات جدید به‌روز شد.\n\n" . "🏷 <b>نام سرویس:</b> <code>{$service['display']}</code>\n" . "📦 <b>حجم جدید:</b> $size گیگابایت\n" . "🗓 <b>تاریخ انقضای جدید:</b> <code>$new_expire_date_jalali</code>\n" . "📡 <b>سرور:</b> $server_display_name\n\n" . "---\n" . "🔗 <b>لینک اتصال (بدون تغییر):</b>\n" . ($type == 'sanaei' ? "<code>$sub</code>" : $sub);
                $msg_response = bot('sendPhoto', ['chat_id' => $user_id, 'photo' => "https://api.qrserver.com/v1/create-qr-code/?data=$encode_url&size=800x800", 'caption' => $caption, 'reply_markup' => $key, 'parse_mode'=>'html']);
                if (isset($msg_response['result']['message_id'])) {
                    $msgid = $msg_response['result']['message_id'];
                    $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                    $stmt_msg->bind_param("is", $msgid, $user_id);
                    $stmt_msg->execute();
                    $stmt_msg->close();
                }
            }
            $stmt_service->close();
        } else { // New service
            $service_name = generateRandomString();
            $expire_time = strtotime("+ $days day");
            $service_id = random_int(100000, 999999);
            $subscribe = '';
            $service_uuid = '';

            if ($type == "marzban") {
                $token = loginPanel($link, $username, $password)['access_token'];
                $proxies = buildProxies($list, 'on');
                $response = createService($service_name, convertToBytes($size . 'GB'), $expire_time, $proxies, $token, $link);
                $create_status = json_decode($response, true);
                $subscribe = (strpos($create_status['subscription_url'], 'http') !== false) ? $create_status['subscription_url'] : $link . $create_status['subscription_url'];
                $service_uuid = $service_name;
            } elseif ($type == "sanaei") {
                include_once("api/sanaei.php");
                login($link, $username, $password);
                $response = addClientToInbound($size, $days, "$link", "$inbound_copy");
                $create_status = json_decode($response, true);
                $service_name = $create_status['remark'];
                $service_uuid = $create_status['uuid'];
                $subscribe = updateVlessLink($server['example_link'], $service_uuid, $service_name);
            }

            if (!empty($subscribe)) {
                $stmt_insert = $connect->prepare("INSERT INTO services (id, uuid,owner,expire,size,name,sub,svname,location,display) VALUES (?,?,?,?,?,?,?,?,?,?)");
                $stmt_insert->bind_param("sssiisssss", $service_id, $service_uuid, $user_id, $expire_time, $size, $service_name, $subscribe, $server['name'], $server_display_name, $service_name);
                $stmt_insert->execute();
                $stmt_insert->close();
                
                $key = json_encode(['inline_keyboard' => [[['text' => "🎛 مدیریت سرویس", 'callback_data' => "service-$service_id"]]]]);
                $final_subscribe_link = apply_custom_domain($subscribe, $server);
                $encode_url = urlencode($final_subscribe_link);
                $caption = "✅ <b>سرویس شما با موفقیت فعال شد</b>\n\n" . " از خرید شما سپاسگزاریم! اطلاعات سرویس جدید شما به شرح زیر است:\n\n" . "🏷 <b>نام سرویس:</b> <code>$service_name</code>\n" . "📦 <b>حجم:</b> $size گیگابایت\n" . "🗓 <b>مدت اعتبار:</b> $days روز\n" . "📡 <b>سرور:</b> $server_display_name\n\n" . "---\n" . "🔗 <b>لینک اتصال:</b>\n" . ($type == 'sanaei' ? "<code>$final_subscribe_link</code>" : $final_subscribe_link);
                $msg_response = bot('sendPhoto', ['chat_id' => $user_id, 'photo' => "https://api.qrserver.com/v1/create-qr-code/?data=$encode_url&size=800x800", 'caption' => $caption, 'reply_markup' => $key, 'parse_mode'=>'html']);
                if (isset($msg_response['result']['message_id'])) {
                    $msgid = $msg_response['result']['message_id'];
                    $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                    $stmt_msg->bind_param("is", $msgid, $user_id);
                    $stmt_msg->execute();
                    $stmt_msg->close();
                }
            }
        }

        if (function_exists('referral_process_successful_order')) {
            referral_process_successful_order($connect, $order);
        }
    } elseif ($action == 'reject') {
        $admin_contact = getMessage($connect, 'support_contact_info'); 
        $message_text = getMessage($connect, 'payment_rejected_user_message', ['size' => $size, 'amount' => $amount, 'admin_contact' => $admin_contact]);
        sendMessage($user_id, $message_text, $start);
        $alert_text = getMessage($connect, 'admin_reject_payment_alert');
        if (!empty($GLOBALS['membercall'])) {
            bot('answercallbackquery', ['callback_query_id' => $GLOBALS['membercall'], 'text' => $alert_text, 'show_alert' => true]);
        }
        if ($messageid) {
            editMessageCaption($chatid, $messageid, "🧾 فیش واریزی رد شد:\n\n👤 کاربر: $user_id\n📦 حجم: $size گیگ\n💸 مبلغ: $amount تومان\n📍 لوکیشن: $server_display_name\n\n❌ وضعیت: رد شده");
        }
        
        $stmt_reject = $connect->prepare("UPDATE orders SET status='rejected' WHERE order_code=?");
        $stmt_reject->bind_param("s", $order_code);
        $stmt_reject->execute();
        $stmt_reject->close();
        // Release any reserved discount usage on rejection
        if (function_exists('release_discount_usage_for_order')) {
            release_discount_usage_for_order($connect, $order_code);
        }
    }
}

function handleTestAccountRequest($connect, $chat_id, $user, $admin_id) {
    if ($user['test'] == "yes" && $chat_id != $admin_id) {
        sendMessage($chat_id, getMessage($connect, 'error_used_test_account'));
        return;
    }
    
    $proccessing_message = sendMessage($chat_id, "⚙️ در حال ساخت اکانت تست... لطفاً چند لحظه صبر کنید.");
    $proccessing_message_id = $proccessing_message['result']['message_id'] ?? null;
    
    $server_res = $connect->query("SELECT * FROM servers WHERE is_test_server = 1 AND status = 'on' LIMIT 1");
    if ($server_res->num_rows == 0) {
        if ($proccessing_message_id) deleteMessage($chat_id, $proccessing_message_id);
        sendMessage($chat_id, getMessage($connect, 'error_test_account_not_available'));
        return;
    }
    $server = $server_res->fetch_assoc();
    $size = 1; $days = 1; $display_name_prefix = "Test";
    $type = $server['type']; $link = $server['link']; $username = $server['username'] ?: 'ندارد'; $password = $server['password'] ?: 'ندارد';
    $inbound_copy = $server['inbound_copy'] ?: 'ندارد'; $server_protocols = $server['protocols']; $server_display_name = $server['display_name'];
    $subscribe = ''; $service_name = ''; $service_uuid = '';
    $service_id = random_int(100000, 999999);
    $expire_time = strtotime("+ $days day");
    $final_display_name = $display_name_prefix . "-" . generateRandomString(4);
    
    if ($type == "marzban") {
        $token = loginPanel($link, $username, $password)['access_token'] ?? null;
        if (!$token) {
            if ($proccessing_message_id) deleteMessage($chat_id, $proccessing_message_id);
            sendMessage($chat_id, "❌ خطا در اتصال به سرور تست. لطفاً به پشتیبانی اطلاع دهید.");
            return;
        }
        $proxies = buildProxies($server_protocols, 'on');
        $service_name = generateRandomString();
        $response = createService($service_name, convertToBytes($size . 'GB'), $expire_time, $proxies, $token, $link);
        $create_status = json_decode($response, true);
        $subscribe = (strpos($create_status['subscription_url'], 'http') !== false) ? $create_status['subscription_url'] : $link . $create_status['subscription_url'];
        $service_uuid = $service_name;
    } elseif ($type == "sanaei") {
        include_once("api/sanaei.php");
        login($link, $username, $password);
        $response = addClientToInbound($size, $days, "$link", "$inbound_copy");
        $create_status = json_decode($response, true);
        $service_name = $create_status['remark'];
        $service_uuid = $create_status['uuid'];
        $subscribe = updateVlessLink($server['example_link'], $service_uuid, $service_name);
    }

    if (!empty($subscribe)) {
        $stmt_insert = $connect->prepare("INSERT INTO services (id, uuid, owner, expire, size, name, sub, svname, location, display) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt_insert->bind_param("sssiisssss", $service_id, $service_uuid, $chat_id, $expire_time, $size, $service_name, $subscribe, $server['name'], $server_display_name, $final_display_name);
        $stmt_insert->execute();
        $stmt_insert->close();
        
        $stmt_update_user = $connect->prepare("UPDATE user SET test='yes' WHERE id=?");
        $stmt_update_user->bind_param("s", $chat_id);
        $stmt_update_user->execute();
        $stmt_update_user->close();

        $key = json_encode(['inline_keyboard' => [[['text' => "🎛 مدیریت سرویس", 'callback_data' => "service-$service_id"]], [['text' => "🛒 خرید سرویس جدید", 'callback_data' => "buy"]]]]);
        $final_subscribe_link = apply_custom_domain($subscribe, $server);
        $encode_url = urlencode($final_subscribe_link);
        $caption = "✅ سرویس تست شما با موفقیت فعال شد!\n\n" . "✨ نام سرویس: $final_display_name\n" . "📦 حجم سرویس: $size گیگ\n" . "⏰ مدت اعتبار: $days روز\n" . "▫️ نوع سرویس: ✨ $server_display_name ✨\n\n" . "🔗 لینک اتصال:\n" . ($type == 'sanaei' ? "<code>$final_subscribe_link</code>" : $final_subscribe_link);
        if ($proccessing_message_id) deleteMessage($chat_id, $proccessing_message_id);
        bot('sendPhoto', ['chat_id' => $chat_id, 'photo' => "https://api.qrserver.com/v1/create-qr-code/?data=$encode_url&size=800x800", 'caption' => $caption, 'reply_markup' => $key, 'parse_mode'=>'html']);
    } else {
        if ($proccessing_message_id) deleteMessage($chat_id, $proccessing_message_id);
        sendMessage($chat_id, "❌ خطایی در ساخت سرویس تست رخ داد. لطفاً به پشتیبانی اطلاع دهید.");
    }
}

// =========================================================================================================
// --- SCRIPT START ---
// =========================================================================================================

$admin_id = get_setting($connect, 'admin_id');
$bot_status = get_setting($connect, 'bot_status') ?? 'on';
$join_status = get_setting($connect, 'join_status') ?? 'off';

$current_user_id = $chat_id ?? $chatid;

// For all interactions (messages and callbacks), make sure the user row exists
if (!empty($current_user_id)) {
    ensure_user_exists($connect, (string)$current_user_id);
}

if ($bot_status === 'off' && $current_user_id != $admin_id) {
    if (isset($data)) bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "🔧 ربات در حال بروزرسانی است.", 'show_alert' => true]);
    else sendMessage($current_user_id, getMessage($connect, 'bot_maintenance_message'));
    exit();
}

// Handle Telegram Stars pre-checkout and successful payments
if (isset($update2['pre_checkout_query'])) {
    bot('answerPreCheckoutQuery', ['pre_checkout_query_id' => $update2['pre_checkout_query']['id'], 'ok' => true]);
    exit();
}
if (isset($update2['message']['successful_payment'])) {
    $payload = $update2['message']['successful_payment']['invoice_payload'] ?? '';
    $payer = $update2['message']['chat']['id'] ?? null;
    if ($payload && $payer) {
        handlePaymentConfirmation($connect, 'approve', $payload, $payer, null);
    }
    exit();
}

if (isset($data)) {
    if($data=="joined"){
        $join_result = joinChannels($connect, $chatid);
        if ($join_result !== true) {
            $channels_list = implode("\n", $join_result);
            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "شما هنوز در تمام کانال‌های زیر عضو نشده‌اید:\n$channels_list", 'show_alert' => true]);
        } else {
            deleteMessage($chatid, $messageid);
            sendMessage($chatid, "✅ عضویت شما تایید گردید.\n\n" . getMessage($connect, 'welcome_message'), $start);
        }
    } elseif (preg_match('/^kyc_(approve|reject)\|(\d+)/', $data, $match)) {
        $action = $match[1];
        $user_id_to_process = $match[2];
        $admin_who_clicked = $update->callback_query->from;

        if ($admin_who_clicked->id == $admin_id) {
            if ($action == 'approve') {
                $stmt = $connect->prepare("UPDATE user SET verification_status = 'verified' WHERE id = ?");
                $stmt->bind_param("s", $user_id_to_process);
                $stmt->execute();
                $stmt->close();
                sendMessage($user_id_to_process, "✅ هویت شما با موفقیت تایید شد.\n\nاکنون می‌توانید خرید خود را با روش کارت به کارت تکمیل کنید. لطفاً به منوی اصلی بازگشته و مجددا مراحل خرید را طی کنید.", $start);
                $admin_name = $admin_who_clicked->first_name;
                editMessageCaption($chatid, $messageid, "✅ <b>تایید شد</b>\nتوسط: $admin_name\nآیدی کاربر: <code>$user_id_to_process</code>", null, 'html');
                bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "✅ هویت کاربر با موفقیت تایید شد."]);
            } elseif ($action == 'reject') {
                $stmt = $connect->prepare("UPDATE user SET verification_status = 'rejected' WHERE id = ?");
                $stmt->bind_param("s", $user_id_to_process);
                $stmt->execute();
                $stmt->close();
                sendMessage($user_id_to_process, "❌ متاسفانه مدارک ارسالی شما جهت احراز هویت، توسط مدیریت رد شد.\n\nدر صورت نیاز می‌توانید با پشتیبانی در ارتباط باشید.", $start);
                $admin_name = $admin_who_clicked->first_name;
                editMessageCaption($chatid, $messageid, "❌ <b>رد شد</b>\nتوسط: $admin_name\nآیدی کاربر: <code>$user_id_to_process</code>", null, 'html');
                bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "❌ درخواست کاربر رد شد."]);
            }
        } else {
            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "⛔️ شما ادمین اصلی نیستید و مجاز به انجام این کار نمی‌باشید.", 'show_alert' => true]);
        }
        exit();
    } elseif (preg_match('/^(approve|reject)\|(.+)/', $data, $matches)) {
        if ($chatid == $admin_id) { handlePaymentConfirmation($connect, $matches[1], $matches[2], $chatid, $messageid); }
    } elseif ($data === 'wallet') {
        // Show wallet menu (callback navigation)
        $stmtb = $connect->prepare("SELECT coin FROM user WHERE id = ? LIMIT 1");
        $stmtb->bind_param("s", $chatid);
        $stmtb->execute();
        $rowb = $stmtb->get_result()->fetch_assoc();
        $stmtb->close();
        $balance_text = number_format((int)($rowb['coin'] ?? 0));
        $msg = "👛 <b>کیف پول شما</b>\n\n💰 موجودی: <b>{$balance_text}</b> تومان";
        $kb = json_encode(['inline_keyboard' => [
            [ ['text' => '➕ شارژ کارت به کارت', 'callback_data' => 'wallet_charge_card'], ['text' => '➕ شارژ رمز ارز', 'callback_data' => 'wallet_charge_crypto'] ],
            [ ['text' => '🔙 بازگشت', 'callback_data' => 'buy'] ]
        ]]);
        editMessageSmart($chatid, $messageid, $msg, $kb, 'html');
        $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
        $stmt_msg->bind_param("is", $messageid, $chatid);
        $stmt_msg->execute();
        $stmt_msg->close();
    } elseif ($data === 'wallet_charge_card') {
        // Ask for amount for card-to-card topup
        $stmt_user = $connect->prepare("UPDATE user SET step='await_wallet_card_amount', tmp=NULL WHERE id=?");
        $stmt_user->bind_param("s", $chatid);
        $stmt_user->execute();
        $stmt_user->close();
        bot('answercallbackquery', ['callback_query_id' => $membercall]);
        editMessageText($chatid, $messageid, "لطفاً مبلغ شارژ کیف پول را به تومان ارسال کنید:", json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => 'wallet']]]]));
    } elseif ($data === 'wallet_charge_crypto') {
        // Ask for amount for crypto topup
        $stmt_user = $connect->prepare("UPDATE user SET step='await_wallet_crypto_amount', tmp=NULL WHERE id=?");
        $stmt_user->bind_param("s", $chatid);
        $stmt_user->execute();
        $stmt_user->close();
        bot('answercallbackquery', ['callback_query_id' => $membercall]);
        editMessageText($chatid, $messageid, "لطفاً مبلغ شارژ کیف پول را به تومان ارسال کنید:", json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => 'wallet']]]]));
    } elseif (preg_match('/^(approve_wallet|reject_wallet)\|([A-Z0-9]+)/', $data, $m)) {
        // Admin approves/rejects wallet topup
        if ($chatid != $admin_id) { bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '⛔️ مجاز نیستید.', 'show_alert' => true]); exit(); }
        $action = $m[1];
        $code = $m[2];
        // ensure table exists
        $connect->query("CREATE TABLE IF NOT EXISTS wallet_topups (id INT AUTO_INCREMENT PRIMARY KEY, code VARCHAR(32) UNIQUE, chat_id BIGINT, amount INT, method ENUM('card','crypto') NOT NULL, status ENUM('pending','approved','rejected') DEFAULT 'pending', created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP)");
        $stmtw = $connect->prepare("SELECT chat_id, amount, status FROM wallet_topups WHERE code = ? LIMIT 1");
        $stmtw->bind_param('s', $code);
        $stmtw->execute();
        $resw = $stmtw->get_result();
        $roww = $resw->fetch_assoc();
        $stmtw->close();
        if (!$roww) { bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '⛔️ رکورد یافت نشد.', 'show_alert' => true]); exit(); }
        if ($roww['status'] !== 'pending') { bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '⛔️ قبلاً پردازش شده.', 'show_alert' => true]); exit(); }
        if ($action === 'approve_wallet') {
            // credit once
            $stmtc = $connect->prepare("UPDATE wallet_topups SET status='approved' WHERE code=?");
            $stmtc->bind_param('s', $code);
            $stmtc->execute();
            $stmtc->close();
            $stmtu = $connect->prepare("UPDATE user SET coin = coin + ? WHERE id = ?");
            $chat_to_credit = (int)$roww['chat_id'];
            $amt = (int)$roww['amount'];
            $stmtu->bind_param('ii', $amt, $chat_to_credit);
            $stmtu->execute();
            $stmtu->close();
            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '✅ تایید شد و کیف پول شارژ گردید.']);
            editMessageCaption($chatid, $messageid, "✅ <b>شارژ کیف پول تایید شد</b>\nکد: <code>{$code}</code>", null, 'html');
            sendMessage($chat_to_credit, "✅ شارژ کیف پول شما به مبلغ " . number_format($amt) . " تومان تایید شد.");
        } else {
            $stmtr = $connect->prepare("UPDATE wallet_topups SET status='rejected' WHERE code=?");
            $stmtr->bind_param('s', $code);
            $stmtr->execute();
            $stmtr->close();
            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '❌ رد شد.']);
            editMessageCaption($chatid, $messageid, "❌ <b>شارژ کیف پول رد شد</b>\nکد: <code>{$code}</code>", null, 'html');
            sendMessage($roww['chat_id'], "❌ شارژ کیف پول شما رد شد. در صورت نیاز با پشتیبانی در ارتباط باشید.");
        }
    }
    elseif (preg_match("/^addelete-plan-(\d+)/", $data, $matches) && $chatid == $admin_id) {
        $plan_id_to_delete = (int)$matches[1];
        
        $stmt_prices = $connect->prepare("DELETE FROM plan_prices WHERE plan_id = ?");
        $stmt_prices->bind_param("i", $plan_id_to_delete);
        $stmt_prices->execute();
        $stmt_prices->close();

        $stmt_plans = $connect->prepare("DELETE FROM plans WHERE id = ?");
        $stmt_plans->bind_param("i", $plan_id_to_delete);
        
        if($stmt_plans->execute()){
            bot('answercallbackquery', ['callback_query_id' => $GLOBALS['membercall'], 'text' => "🗑 تعرفه با موفقیت حذف شد.", 'show_alert' => false]);
            $plans_res = $connect->query("SELECT * FROM plans ORDER BY id ASC");
            if ($plans_res->num_rows > 0) {
                $keyboard = ['inline_keyboard' => []];
                while($plan = $plans_res->fetch_assoc()){
                    $keyboard['inline_keyboard'][] = [['text' => "{$plan['title']} ({$plan['volume']}GB - {$plan['duration']} روز) 🗑", 'callback_data' => "addelete-plan-{$plan['id']}"]];
                }
                editMessageText($chatid, $messageid, "لیست تعرفه‌ها بروزرسانی شد. برای حذف، روی تعرفه مورد نظر کلیک کنید:", json_encode($keyboard));
            } else {
                editMessageText($chatid, $messageid, "تمام تعرفه‌ها حذف شدند.");
            }
        } else {
            bot('answercallbackquery', ['callback_query_id' => $GLOBALS['membercall'], 'text' => "❌ خطا در حذف تعرفه.", 'show_alert' => true]);
        }
        $stmt_plans->close();
    }
    elseif (preg_match("/^addelete-(\d+)/", $data, $matches) && $chatid == $admin_id) {
        $service_id_to_delete = $matches[1];
        bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "🗑 سرویس $service_id_to_delete حذف خواهد شد.", 'show_alert' => true]);
        deleteMessage($chatid, $messageid);
    }
    elseif ($chatid == $admin_id && preg_match('/^manage_server\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $stmt = $connect->prepare("SELECT * FROM servers WHERE id = ? LIMIT 1");
        $stmt->bind_param("i", $server_id);
        $stmt->execute();
        $server_res = $stmt->get_result();
        if($server_res->num_rows > 0){
            $server = $server_res->fetch_assoc();
            $status_text = ($server['status'] == 'on') ? 'خاموش کردن 🔴' : 'روشن کردن 🟢';
            $hide_icon = $server['is_hidden'] ? '👁️' : '🙈';
            $test_icon = $server['is_test_server'] ? '🧪' : '⚪️';
            
            $keyboard = json_encode(['inline_keyboard' => [
                [['text' => $status_text, 'callback_data' => "toggle_server|$server_id"]],
                [['text' => $hide_icon, 'callback_data' => "toggle_hide_server|$server_id"], ['text' => $test_icon, 'callback_data' => "set_test_server|$server_id"], ['text' => '🗑', 'callback_data' => "delete_server|$server_id"]],
                [['text' => '🔙 بازگشت به لیست سرورها', 'callback_data' => 'back_to_servers_list']]
            ]]);
            $text = "⚙️ مدیریت سرور: <b>{$server['display_name']}</b>\n\n- $hide_icon : نمایش/مخفی کردن از لیست خرید\n- $test_icon : انتخاب به عنوان سرور تست";
            editMessageText($chatid, $messageid, $text, $keyboard, "html");
        } else {
            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '❌ سرور یافت نشد.', 'show_alert' => true]);
        }
        $stmt->close();
    }
    elseif ($chatid == $admin_id && $data == 'back_to_servers_list') {
        $result = $connect->query("SELECT * FROM servers ORDER BY id ASC");
        $keyboard = ['inline_keyboard' => []];
        while ($row = $result->fetch_assoc()) {
            $status_icon = ($row['status'] == 'on') ? "🟢" : "🔴";
            $hide_icon = $row['is_hidden'] ? '🙈' : '';
            $test_icon = $row['is_test_server'] ? '🧪' : '';
            $keyboard['inline_keyboard'][] = [['text' => "$status_icon {$row['display_name']} $hide_icon $test_icon", 'callback_data' => "manage_server|{$row['id']}"]];
        }
        $text = "🎛️ <b>مدیریت سرورها</b>\n\nبرای مدیریت هر سرور، روی نام آن کلیک کنید.";
        editMessageText($chatid, $messageid, $text, json_encode($keyboard), "html");
    }
    elseif ($chatid == $admin_id && preg_match('/^toggle_hide_server\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $stmt = $connect->prepare("UPDATE servers SET is_hidden = 1 - is_hidden WHERE id = ?");
        $stmt->bind_param("i", $server_id);
        $stmt->execute();
        $stmt->close();
        bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "✅ وضعیت نمایش سرور تغییر کرد.", 'show_alert' => false]);
        $data = "manage_server|$server_id"; // Refresh
    }
    elseif ($chatid == $admin_id && preg_match('/^set_test_server\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $connect->query("UPDATE servers SET is_test_server = 0"); // Reset all
        $stmt = $connect->prepare("UPDATE servers SET is_test_server = 1 WHERE id = ?");
        $stmt->bind_param("i", $server_id);
        $stmt->execute();
        $stmt->close();
        bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "✅ این سرور به عنوان سرور تست انتخاب شد.", 'show_alert' => false]);
        $data = "manage_server|$server_id"; // Refresh
    }

    if ($chatid == $admin_id && preg_match('/^toggle_server\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $stmt = $connect->prepare("SELECT status FROM servers WHERE id = ? LIMIT 1");
        $stmt->bind_param("i", $server_id);
        $stmt->execute();
        $server_res = $stmt->get_result();
        if($server_res->num_rows > 0){
            $server = $server_res->fetch_assoc();
            $new_status = ($server['status'] == 'on') ? 'off' : 'on';
            $stmt_update = $connect->prepare("UPDATE servers SET status = ? WHERE id = ?");
            $stmt_update->bind_param("si", $new_status, $server_id);
            $stmt_update->execute();
            $stmt_update->close();
            
            $new_status_alert = ($new_status == 'on') ? 'روشن' : 'خاموش';
            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "✅ سرور با موفقیت $new_status_alert شد.", 'show_alert' => false]);
            $data = "manage_server|$server_id"; // Refresh
        }
        $stmt->close();
    }
    elseif ($chatid == $admin_id && preg_match('/^delete_server\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $keyboard = json_encode(['inline_keyboard' => [
            [['text' => '✅ بله، حذف کن', 'callback_data' => "confirm_delete_server|$server_id"], ['text' => '❌ نه، لغو', 'callback_data' => "manage_server|$server_id"]]
        ]]);
        $text = "⚠️ <b>آیا از حذف این سرور مطمئن هستید؟</b>\nاین عمل غیرقابل بازگشت است!";
        editMessageText($chatid, $messageid, $text, $keyboard, "html");
    }
    elseif ($chatid == $admin_id && preg_match('/^confirm_delete_server\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $stmt = $connect->prepare("DELETE FROM servers WHERE id = ?");
        $stmt->bind_param("i", $server_id);
        $stmt->execute();
        $stmt->close();
        bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '🗑 سرور با موفقیت حذف شد.', 'show_alert' => true]);
        
        $result = $connect->query("SELECT * FROM servers ORDER BY id ASC");
        $keyboard = ['inline_keyboard' => []];
        while ($row = $result->fetch_assoc()) {
            $status_icon = ($row['status'] == 'on') ? "🟢" : "🔴";
            $hide_icon = $row['is_hidden'] ? '🙈' : '';
            $test_icon = $row['is_test_server'] ? '🧪' : '';
            $keyboard['inline_keyboard'][] = [['text' => "$status_icon {$row['display_name']} $hide_icon $test_icon", 'callback_data' => "manage_server|{$row['id']}"]];
        }
        $text = "🎛️ <b>مدیریت سرورها</b>\n\nلیست سرورها بروزرسانی شد.";
        editMessageText($chatid, $messageid, $text, json_encode($keyboard), "html");
    }
    elseif ($chatid == $admin_id && preg_match('/^set_price_for_server\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $stmt_srv = $connect->prepare("SELECT * FROM servers WHERE id = ?"); $stmt_srv->bind_param("i", $server_id); $stmt_srv->execute(); $server = $stmt_srv->get_result()->fetch_assoc(); $stmt_srv->close();
        $sql = "SELECT p.id as plan_id, p.title, p.volume, p.duration, pp.price, pp.stars_price FROM plans p LEFT JOIN plan_prices pp ON p.id = pp.plan_id AND pp.server_id = ? ORDER BY p.id ASC";
        $stmt = $connect->prepare($sql);
        $stmt->bind_param("i", $server_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
        $keyboard = ['inline_keyboard' => []];
        while($plan = $result->fetch_assoc()){
            $parts = [];
            if (isset($plan['price']) && $plan['price'] > 0) $parts[] = number_format($plan['price']) . ' تومان';
            if (isset($plan['stars_price']) && $plan['stars_price'] > 0) $parts[] = $plan['stars_price'] . '⭐';
            $price_text = $parts ? implode(' - ', $parts) : 'قیمت‌گذاری نشده';
            $keyboard['inline_keyboard'][] = [['text' => "{$plan['title']} - $price_text", 'callback_data' => "select_plan_for_price|{$server_id}|{$plan['plan_id']}"]];
        }
        $keyboard['inline_keyboard'][] = [['text' => '🔙 بازگشت', 'callback_data' => 'back_to_pricing_main']];
        $text = "لطفاً تعرفه‌ای که می‌خواهید برای سرور <b>{$server['display_name']}</b> قیمت‌گذاری کنید را انتخاب نمایید:";
        editMessageText($chatid, $messageid, $text, json_encode($keyboard), "html");
        $stmt->close();
    }
    elseif ($chatid == $admin_id && $data == 'back_to_pricing_main') {
        $servers_res = $connect->query("SELECT * FROM servers WHERE status = 'on' AND is_hidden = 0 ORDER BY id ASC");
        $keyboard = ['inline_keyboard' => []];
        while($server = $servers_res->fetch_assoc()) {
            $keyboard['inline_keyboard'][] = [['text' => $server['display_name'], 'callback_data' => "set_price_for_server|{$server['id']}"]];
        }
        $keyboard['inline_keyboard'][] = [['text' => "🔙 بازگشت به پنل", 'callback_data' => "admin_panel_main"]];
        editMessageText($chatid, $messageid, "لطفاً سروری که می‌خواهید قیمت‌های آن را تنظیم کنید، انتخاب نمایید:", json_encode($keyboard));
    }
    elseif ($chatid == $admin_id && preg_match('/^select_plan_for_price\|(\d+)\|(\d+)/', $data, $matches)) {
        $server_id = $matches[1];
        $plan_id = $matches[2];
        $stmt = $connect->prepare("UPDATE user SET step=?, msgid=? WHERE id=?");
        $step = "set_price|{$server_id}|{$plan_id}";
        $stmt->bind_param("sis", $step, $messageid, $chatid);
        $stmt->execute();
        $stmt->close();
        bot('answercallbackquery', ['callback_query_id' => $membercall]);
        editMessageText($chatid, $messageid, "لطفا قیمت جدید را برای این پلن به تومان وارد کنید.\n\n(برای حذف قیمت، عدد <code>0</code> را ارسال کنید)");
    }
    elseif($data =="kankel"){
        deleteMessage($chatid, $user2['msgid']);
        $stmt = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
        $stmt->bind_param("s", $chatid);
        $stmt->execute();
        $stmt->close();
        sendMessage($chatid, getMessage($connect, 'welcome_message'), $start);   
    }
    elseif($data =="no" || $data == "no_op" || $data == "admin_panel_main"){
        bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "این دکمه عملیاتی ندارد.", 'show_alert' => false]);
    }
    elseif(preg_match('/^qr/',$data)){
        $final_link = apply_custom_domain_by_link($connect, $user2['qr']);
        $text = urlencode($final_link);
        sendphoto($chatid,"https://chart.googleapis.com/chart?chs=300x300&cht=qr&chl=$text&choe=UTF-8");
    }
    elseif($data=="getc"){
        $vmess = apply_custom_domain_by_link($connect, $user2['qr']);
        bot('sendMessage',['chat_id'=>$chatid, 'text'=>"🔗 لینک اتصال سرویس شما:\n\n<code>$vmess</code>", 'parse_mode' => "html"]);
    }
    elseif($data=="buy"){
        if (($user2['step'] ?? '') === 'waiting_payment_stars' && !empty($user2['tmp'])) {
            $parts = explode('|', $user2['tmp']);
            if (count($parts) > 1) {
                $inv_msg_id = (int)$parts[1];
                deleteMessage($chatid, $inv_msg_id);
            }
        }
        $stmt = $connect->prepare("UPDATE user SET step='none', tmp=NULL WHERE id=?");
        $stmt->bind_param("s", $chatid);
        $stmt->execute();
        $stmt->close();
        $keyboard = ['inline_keyboard' => []];
        $result = $connect->query("SELECT * FROM servers WHERE status='on' AND is_hidden = 0 AND is_test_server = 0 ORDER BY id ASC");
        while ($row = $result->fetch_assoc()) {
            $keyboard['inline_keyboard'][] = [['text' => "✨ {$row['display_name']} ✨", 'callback_data' => "selectserver|{$row['id']}"]];
        }
        // If the callback originated from a media message (payment photo), delete it
        $came_from_media = isset($update2['callback_query']['message']['photo'])
                           || isset($update2['callback_query']['message']['video'])
                           || isset($update2['callback_query']['message']['document'])
                           || isset($update2['callback_query']['message']['animation'])
                           || isset($update2['callback_query']['message']['audio']);

        if ($came_from_media) {
            // Remove the big image and send a fresh message
            deleteMessage($chatid, $messageid);
            if (empty($keyboard['inline_keyboard'])) {
                $resp = sendMessage($chatid, getMessage($connect, 'error_no_active_server_for_buy'));
            } else {
                $resp = sendMessage($chatid, getMessage($connect, 'ask_location'), json_encode($keyboard));
            }
            $new_id = $resp['result']['message_id'] ?? null;
            if ($new_id) {
                $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                $stmt_msg->bind_param("is", $new_id, $chatid);
                $stmt_msg->execute();
                $stmt_msg->close();
            }
        } else {
            if (empty($keyboard['inline_keyboard'])) {
                editMessageText($chatid, $messageid, getMessage($connect, 'error_no_active_server_for_buy'));
            } else {
                editMessageSmart($chatid, $messageid, getMessage($connect, 'ask_location'), json_encode($keyboard));
            }
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
    }
    elseif (preg_match("/^tariffs_for_server\|(\d+)/", $data, $matches)) {
        $server_id = $matches[1];
        $stmt_server = $connect->prepare("SELECT display_name FROM servers WHERE id=? LIMIT 1");
        $stmt_server->bind_param("i", $server_id);
        $stmt_server->execute();
        $server_res = $stmt_server->get_result();
        if($server_res->num_rows == 0) { bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => '⛔️ سرور یافت نشد!']); exit(); }
        $server_display_name = $server_res->fetch_assoc()['display_name'];
        $stmt_server->close();
        $sql = "SELECT p.title, p.volume, p.duration, pp.price, pp.stars_price FROM plans p JOIN plan_prices pp ON p.id = pp.plan_id WHERE pp.server_id = ? AND ((pp.price IS NOT NULL AND pp.price > 0) OR (pp.stars_price IS NOT NULL AND pp.stars_price > 0)) ORDER BY p.id ASC";
        $stmt = $connect->prepare($sql); $stmt->bind_param("i", $server_id); $stmt->execute();
        $plans_res = $stmt->get_result();
        if ($plans_res->num_rows > 0) {
            $msg = "💎 <b>تعرفه‌های سرور $server_display_name</b>\n\n";
            while ($plan = $plans_res->fetch_assoc()) {
                $msg .= "🔰 <b>{$plan['title']}</b>\n   - 📦 حجم: {$plan['volume']} گیگابایت\n   - ⏳ مدت: {$plan['duration']} روز";
                if ($plan['price'] > 0) {
                    $msg .= "\n   - 💸 قیمت: <b>" . number_format($plan['price']) . "</b> تومان";
                }
                if ($plan['stars_price'] > 0) {
                    $msg .= "\n   - 🌟 استارز: <b>{$plan['stars_price']}</b>";
                }
                $msg .= "\n\n";
            }
            $key = json_encode(['inline_keyboard' => [[['text' => '🛒 خرید سرویس از این لوکیشن', 'callback_data' => "selectserver|$server_id"]]]]);
            editMessageText($chatid, $messageid, $msg, $key, "html");
        } else {
            editMessageText($chatid, $messageid, getMessage($connect, 'error_no_tariff_for_server', ['server_name' => $server_display_name]), $start, "html");
        }
        $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
        $stmt_msg->bind_param("is", $messageid, $chatid);
        $stmt_msg->execute();
        $stmt_msg->close();
        $stmt->close();
    }
    elseif (preg_match("/^selectserver\|(\d+)/", $data, $matches)) {
        $server_id = $matches[1];
        $stmt_server = $connect->prepare("SELECT name FROM servers WHERE id=? AND status='on' LIMIT 1");
        $stmt_server->bind_param("i", $server_id);
        $stmt_server->execute();
        $server_res = $stmt_server->get_result();
        if ($server = $server_res->fetch_assoc()) {
            $back = ($user2['step'] == "exten") ? "ex-" . $user2['want'] : "buy";
        $sql = "SELECT p.id, p.title, pp.price, pp.stars_price FROM plans p JOIN plan_prices pp ON p.id = pp.plan_id WHERE pp.server_id = ? AND ((pp.price IS NOT NULL AND pp.price > 0) OR (pp.stars_price IS NOT NULL AND pp.stars_price > 0)) ORDER BY p.id ASC";
            $stmt = $connect->prepare($sql); $stmt->bind_param("i", $server_id); $stmt->execute();
            $plans_res = $stmt->get_result();
            $keyboard = ['inline_keyboard' => []];
            if ($plans_res->num_rows > 0) {
                while ($plan = $plans_res->fetch_assoc()) {
                    $price_text = [];
                    if ($plan['price'] > 0) $price_text[] = number_format($plan['price']) . ' تومان';
                    if ($plan['stars_price'] > 0) $price_text[] = $plan['stars_price'] . '⭐';
                    $text_price = implode(' - ', $price_text);
                    $keyboard['inline_keyboard'][] = [['text' => "{$plan['title']} - $text_price", 'callback_data' => "plan|{$server['name']}|{$plan['id']}|{$plan['price']}|{$plan['stars_price']}"]];
                }
            } else {
                 $keyboard['inline_keyboard'][] = [['text' => "❌ تعرفه‌ای برای این سرور تعریف نشده", 'callback_data' => "no_op"]];
            }
            $keyboard['inline_keyboard'][] = [['text' => "🔙 بازگشت", 'callback_data' => "$back"]];
            editMessageText($chatid, $messageid, getMessage($connect, 'ask_plan'), json_encode($keyboard));
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
            $stmt->close();
        }
        $stmt_server->close();
    }
    elseif (preg_match("/^plan\|(.+?)\|(\d+)\|(\d+)\|(\d+)/", $data, $matches)) {
        $loc = $matches[1]; $id = (int)$matches[2]; $amount = (int)$matches[3]; $stars = (int)$matches[4];
        $stmt_server = $connect->prepare("SELECT display_name FROM servers WHERE name=? AND status='on' LIMIT 1");
        $stmt_server->bind_param("s", $loc);
        $stmt_server->execute();
        $server_check = $stmt_server->get_result();
        
        $stmt_plan = $connect->prepare("SELECT * FROM plans WHERE id=? LIMIT 1");
        $stmt_plan->bind_param("i", $id);
        $stmt_plan->execute();
        $plan_check = $stmt_plan->get_result();
        
        if ($server_check->num_rows > 0 && $plan = $plan_check->fetch_assoc()) {
            $server = $server_check->fetch_assoc();
            $key = json_encode(['inline_keyboard' => [[['text' => "تایید و پرداخت", 'callback_data' => "confirm|$loc-$id-$amount-$stars"], ['text' => "انصراف", 'callback_data' => "kankel"]]]]);
            $msg = getMessage($connect, 'confirm_purchase_details', ['size' => $plan['volume'], 'location' => $server['display_name'], 'amount' => number_format($amount)]);
            if ($stars > 0) { $msg .= "\n🌟 تعداد استارز: <b>{$stars}</b>"; }
            editMessageText($chatid, $messageid, $msg, $key, 'html');
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
        $stmt_server->close();
        $stmt_plan->close();
    }
    elseif (preg_match('/^confirm\|(.+)/', $data, $matches)) {
        list($loc, $id, $amount, $stars) = explode("-", $matches[1]);
        
        $stmt_plan = $connect->prepare("SELECT * FROM plans WHERE id=? LIMIT 1");
        $stmt_plan->bind_param("i", $id);
        $stmt_plan->execute();
        $getplan = $stmt_plan->get_result();
        
        $stmt_server = $connect->prepare("SELECT display_name FROM servers WHERE name=? AND status='on' LIMIT 1");
        $stmt_server->bind_param("s", $loc);
        $stmt_server->execute();
        $server_check = $stmt_server->get_result();

        if ($getplan->num_rows > 0 && $server_check->num_rows > 0) {
            $plan = $getplan->fetch_assoc();
            $server = $server_check->fetch_assoc();
            $order_code = strtoupper(substr(md5(uniqid()), 0, 8));
            
            $stmt_insert = $connect->prepare("INSERT INTO orders (user_id, order_code, location, size, amount, stars_amount, days, type, svcode, base_amount) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $type = ($user2['step'] == "exten") ? 'exten' : 'new';
            $svcode = ($user2['step'] == "exten") ? $user2['want'] : null;
            $stmt_insert->bind_param("sssiiisssi", $chatid, $order_code, $loc, $plan['volume'], $amount, $stars, $plan['duration'], $type, $svcode, $amount);
            $stmt_insert->execute();
            $stmt_insert->close();

            $stmt_user = $connect->prepare("UPDATE user SET step='choose_gateway', tmp=? WHERE id=?");
            $stmt_user->bind_param("ss", $order_code, $chatid);
            $stmt_user->execute();
            $stmt_user->close();
            
            $keyboard = ['inline_keyboard' => []];
            $card_payment_status = get_setting($connect, 'card_payment_status') ?? 'on';
            $crypto_payment_status = get_setting($connect, 'crypto_payment_status') ?? 'off';
            $stars_payment_status = get_setting($connect, 'stars_payment_status') ?? 'off';
            
            if($card_payment_status == 'on'){
                $keyboard['inline_keyboard'][] = [['text' => '💳 کارت به کارت', 'callback_data' => "gateway|card|$order_code"]];
            }
            if($crypto_payment_status == 'on'){
                $keyboard['inline_keyboard'][] = [['text' => '💎 پرداخت با رمز ارز', 'callback_data' => "gateway|crypto|$order_code"]];
            }
            if($stars_payment_status == 'on' && $stars > 0){
                $keyboard['inline_keyboard'][] = [['text' => '🌟 پرداخت با استارز', 'callback_data' => "gateway|stars|$order_code"]];
            }
            // کیف پول
            $keyboard['inline_keyboard'][] = [['text' => 'پرداخت از کیف پول', 'callback_data' => "gateway|wallet|$order_code"]];

            if(empty($keyboard['inline_keyboard'])){
                 editMessageText($chatid, $messageid, "⛔️ در حال حاضر هیچ روش پرداختی فعال نیست. لطفاً با پشتیبانی تماس بگیرید.");
            } else {
                $msg = getMessage($connect, 'ask_payment_method', ['size' => $plan['volume'], 'amount' => number_format($amount), 'location' => $server['display_name']]);
                editMessageText($chatid, $messageid, $msg, json_encode($keyboard));
            }
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
        $stmt_plan->close();
        $stmt_server->close();
    } elseif(preg_match('/^gateway\|(card|crypto|wallet|stars)\|(.+)/', $data, $match)){
        $gateway_type = $match[1];
        $order_code = $match[2];
        $stmt = $connect->prepare("SELECT * FROM orders WHERE order_code = ? LIMIT 1");
        $stmt->bind_param("s", $order_code);
        $stmt->execute();
        $order_res = $stmt->get_result();
        if ($order_res->num_rows == 0) {
            editMessageText($chatid, $messageid, "خطا: سفارش شما یافت نشد.", $start);
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
            exit();
        }
        $order = $order_res->fetch_assoc();
        $stmt->close();

        if($gateway_type == 'card'){
            $kyc_status = get_setting($connect, 'kyc_status');
            
            if($kyc_status == 'on' && $user2['verification_status'] != 'verified'){
                $stmt_user = $connect->prepare("UPDATE user SET step='pending_kyc_start', tmp=? WHERE id=?");
                $stmt_user->bind_param("ss", $order_code, $chatid);
                $stmt_user->execute();
                $stmt_user->close();
                $keyboard = json_encode(['inline_keyboard' => [[['text' => '✅ شروع احراز هویت', 'callback_data' => "start_kyc"]]]]);
                editMessageText($chatid, $messageid, "⚠️ <b>نیاز به احراز هویت</b>\n\nبرای استفاده از روش پرداخت کارت به کارت، ابتدا باید هویت خود را تایید کنید. این کار فقط یک بار انجام می‌شود.", $keyboard, "html");
                $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                $stmt_msg->bind_param("is", $messageid, $chatid);
                $stmt_msg->execute();
                $stmt_msg->close();
                exit();
            }

            // Ask for optional discount code before showing card info
            $stmt_user = $connect->prepare("UPDATE user SET step='awaiting_discount_code', tmp=? WHERE id=?");
            $stmt_user->bind_param("ss", $order_code, $chatid);
            $stmt_user->execute();
            $stmt_user->close();
            $skip_key = json_encode(['inline_keyboard' => [[['text' => 'بدون کد', 'callback_data' => "skip_discount|$order_code"]]]]);
            $amount_t = number_format($order['amount']);
            $prompt = "🎟️ آیا کد تخفیف دارید؟\n\nاگر کد دارید، همینجا ارسال کنید.\nدر غیر این صورت روی دکمه «بدون کد» بزنید.\n\nمبلغ فعلی: {$amount_t} تومان";
            editMessageText($chatid, $messageid, $prompt, $skip_key, "html");
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();

        } elseif($gateway_type == 'crypto'){
            $wallets = get_setting($connect, 'crypto_wallets');
            if(empty($wallets)){
                 editMessageText($chatid, $messageid, "⛔️ هیچ آدرس کیف پولی در ربات تنظیم نشده است. لطفاً با پشتیبانی تماس بگیرید.", $start);
                 $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                 $stmt_msg->bind_param("is", $messageid, $chatid);
                 $stmt_msg->execute();
                 $stmt_msg->close();
                 exit();
            }

            $usdt_price = get_usdt_price();
            $amount_toman = $order['amount'];
            $amount_usdt = '';
            if($usdt_price){
                // Calculate USDT amount with a 5% markup so the user knows how much to send
                $usdt_amount_val = round(($amount_toman / $usdt_price) * 1.05, 2);
                $amount_usdt = " (حدود <code>{$usdt_amount_val}</code> تتر)";
            }

            $stmt_user = $connect->prepare("UPDATE user SET step='waiting_payment_crypto', tmp=? WHERE id=?");
            $stmt_user->bind_param("ss", $order_code, $chatid);
            $stmt_user->execute();
            $stmt_user->close();

            $message_text = getMessage($connect, 'crypto_payment_message', [
                'amount_toman' => number_format($amount_toman),
                'amount_usdt' => $amount_usdt,
                'wallets' => $wallets
            ]);
            $back_inline = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => 'buy']]]]);
            editMessageText($chatid, $messageid, $message_text, $back_inline, "html");
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        } elseif($gateway_type == 'stars') {
            $stars_needed = (int)($order['stars_amount'] ?? 0);
            if($stars_needed <= 0){
                editMessageText($chatid, $messageid, '⛔️ این سفارش قابل پرداخت با استارز نیست.', $start);
                $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                $stmt_msg->bind_param("is", $messageid, $chatid);
                $stmt_msg->execute();
                $stmt_msg->close();
                exit();
            }

            $invoice_res = bot('sendInvoice', [
                'chat_id' => $chatid,
                'title' => 'پرداخت سفارش',
                'description' => 'پرداخت سفارش ' . $order_code,
                'payload' => $order_code,
                'currency' => 'XTR',
                'prices' => json_encode([['label' => 'Order', 'amount' => $stars_needed]])
            ]);
            $invoice_msg_id = $invoice_res['result']['message_id'] ?? null;
            $tmp_val = $invoice_msg_id ? ($order_code . '|' . $invoice_msg_id) : $order_code;
            $stmt_user = $connect->prepare("UPDATE user SET step='waiting_payment_stars', tmp=? WHERE id=?");
            $stmt_user->bind_param("ss", $tmp_val, $chatid);
            $stmt_user->execute();
            $stmt_user->close();

            $back_inline = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => 'buy']]]]);
            $message_text = getMessage($connect, 'stars_payment_message');
            editMessageText($chatid, $messageid, $message_text, $back_inline, 'html');
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $messageid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        } elseif($gateway_type == 'wallet') {
            // Wallet payment flow
            // Fetch user's balance
            $stmt_userb = $connect->prepare("SELECT coin FROM user WHERE id = ? LIMIT 1");
            $stmt_userb->bind_param("s", $chatid);
            $stmt_userb->execute();
            $user_row = $stmt_userb->get_result()->fetch_assoc();
            $stmt_userb->close();

            $user_balance = (int)($user_row['coin'] ?? 0);
            $amount_need = (int)$order['amount'];

            if (($order['status'] ?? '') === 'approved') {
                bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => 'این سفارش قبلا پرداخت شده است.', 'show_alert' => true]);
                exit();
            }

            if ($user_balance < $amount_need) {
                $need_text = number_format($amount_need);
                $bal_text = number_format($user_balance);
                $msg = "موجودی کیف پول شما کافی نیست.\n\nمبلغ سفارش: <b>{$need_text}</b> تومان\nموجودی فعلی: <b>{$bal_text}</b> تومان";
                $kb = json_encode(['inline_keyboard' => [[['text' => '➕ شارژ کیف پول', 'callback_data' => "charge_wallet|$order_code"]], [['text' => 'بازگشت', 'callback_data' => 'buy']]]]);
                editMessageText($chatid, $messageid, $msg, $kb, 'html');
                $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                $stmt_msg->bind_param("is", $messageid, $chatid);
                $stmt_msg->execute();
                $stmt_msg->close();
                exit();
            }

            // Attempt atomic deduction
            $stmt_deduct = $connect->prepare("UPDATE user SET coin = coin - ? WHERE id = ? AND coin >= ?");
            $stmt_deduct->bind_param("isi", $amount_need, $chatid, $amount_need);
            $stmt_deduct->execute();
            $affected = $stmt_deduct->affected_rows;
            $stmt_deduct->close();

            if ($affected <= 0) {
                bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => 'کسر موجودی انجام نشد. لطفا مجددا تلاش کنید.', 'show_alert' => true]);
                exit();
            }

            // Approve order
            $stmt_update_order = $connect->prepare("UPDATE orders SET status='approved' WHERE order_code=?");
            $stmt_update_order->bind_param("s", $order_code);
            $stmt_update_order->execute();
            $stmt_update_order->close();
            if (function_exists('confirm_discount_usage_for_order')) { confirm_discount_usage_for_order($connect, $order_code); }

            // Load server
            $stmt_server = $connect->prepare("SELECT * FROM servers WHERE name = ? AND status = 'on' LIMIT 1");
            $stmt_server->bind_param("s", $order['location']);
            $stmt_server->execute();
            $server_res = $stmt_server->get_result();
            if ($server_res->num_rows == 0) {
                $stmt_server->close();
                // refund
                $stmt_refund = $connect->prepare("UPDATE user SET coin = coin + ? WHERE id = ?");
                $stmt_refund->bind_param("is", $amount_need, $chatid);
                $stmt_refund->execute();
                $stmt_refund->close();
                bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => 'سرور در دسترس نیست. مبلغ به کیف پول بازگشت.', 'show_alert' => true]);
                exit();
            }
            $server = $server_res->fetch_assoc();
            $stmt_server->close();

            $type = $server['type'];
            $link = $server['link'];
            $username = $server['username'] ?: '';
            $password = $server['password'] ?: '';
            $inbound_copy = $server['inbound_copy'] ?: '';
            $list = $server['protocols'];
            $server_display_name = $server['display_name'];

            $size = (int)$order['size'];
            $days = (int)$order['days'];
            $user_id = $order['user_id'];
            $amount_fmt = number_format($order['amount']);

            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => 'در حال پردازش سفارش...', 'show_alert' => false]);

            if ($order['type'] === 'exten') {
                    $stmt_service = $connect->prepare("SELECT * FROM services WHERE id = ? LIMIT 1");
                    $stmt_service->bind_param("s", $order['svcode']);
                    $stmt_service->execute();
                    $service_res = $stmt_service->get_result();
                    if($service_res->num_rows > 0){
                        $service = $service_res->fetch_assoc();
                        $name = $service['name'];
                        $current_expire_time = (int)($service['expire'] ?? 0);
                        $base_time = $current_expire_time > time() ? $current_expire_time : time();
                        $days_to_extend = max(0, (int)$days);
                        $new_expire_time = $days_to_extend > 0 ? strtotime("+{$days_to_extend} day", $base_time) : $base_time;

                        include_once('jdf.php');
                        $Y_new = date('Y', $new_expire_time);
                        $m_new = date('m', $new_expire_time);
                        $d_new = date('d', $new_expire_time);
                        $new_expire_date_jalali = gregorian_to_jalali($Y_new, $m_new, $d_new, '/');

                    if ($type == "marzban") {
                        $token = loginPanel($link, $username, $password)['access_token'] ?? null;
                        if ($token) {
                            $proxies = buildProxies($list, 'on');
                            resetUser($name, $token, $link);
                            tamdid($name, convertToBytes($size . 'GB'), $new_expire_time, $proxies, $token, $link);
                        }
                    } elseif ($type == "sanaei") {
                        include_once("api/sanaei.php");
                        login($link, $username, $password);
                        $uuid = $service['uuid'];
                        ResetClientTraffic($name, $inbound_copy, $link);
                        UpdateClient($uuid, $name, $size, $new_expire_time * 1000, "$link", "$inbound_copy");
                    }

                    $stmt_update_service = $connect->prepare("UPDATE `services` SET size=?, expire=?, volume_notified='0' WHERE id=?");
                    $stmt_update_service->bind_param("iis", $size, $new_expire_time, $order['svcode']);
                    $stmt_update_service->execute();
                    $stmt_update_service->close();

                    $key = json_encode(['inline_keyboard' => [[['text' => "🔎 مشاهده سرویس", 'callback_data' => "service-" . $order['svcode']]]]]);
                    $sub = apply_custom_domain($service['sub'], $server);
                    $encode_url = urlencode($sub);
                    $caption = "✅ <b>تمدید سرویس با موفقیت انجام شد</b>\n\n" . "👤 <b>کاربر:</b> <code>{$user_id}</code>\n" . "💾 <b>حجم:</b> {$size} گیگ\n" . "💰 <b>مبلغ:</b> {$amount_fmt} تومان\n" . "📍 <b>سرور:</b> {$server_display_name}\n\n" . "---\n" . "🧩 <b>لینک اشتراک (برای کپی/افزودن):</b>\n" . ($type == 'sanaei' ? "<code>$sub</code>" : $sub);
                    $msg_response = bot('sendPhoto', ['chat_id' => $user_id, 'photo' => "https://api.qrserver.com/v1/create-qr-code/?data=$encode_url&size=800x800", 'caption' => $caption, 'reply_markup' => $key, 'parse_mode'=>'html']);
                    if (isset($msg_response['result']['message_id'])) {
                        $msgid2 = $msg_response['result']['message_id'];
                        $stmt_msg2 = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                        $stmt_msg2->bind_param("is", $msgid2, $user_id);
                        $stmt_msg2->execute();
                        $stmt_msg2->close();
                    }
                }
                $stmt_service->close();
                editMessageText($chatid, $messageid, '✅ تمدید سفارش با کیف پول انجام شد.', null, 'html');
            } else {
                $service_name = generateRandomString();
                $expire_time = strtotime("+ $days day");
                $service_id = random_int(100000, 999999);
                $subscribe = '';
                $service_uuid = '';

                if ($type == "marzban") {
                    $token = loginPanel($link, $username, $password)['access_token'] ?? null;
                    if ($token) {
                        $proxies = buildProxies($list, 'on');
                        $response = createService($service_name, convertToBytes($size . 'GB'), $expire_time, $proxies, $token, $link);
                        $create_status = json_decode($response, true);
                        $subscribe = (strpos($create_status['subscription_url'] ?? '', 'http') !== false) ? $create_status['subscription_url'] : ($link . ($create_status['subscription_url'] ?? ''));
                        $service_uuid = $service_name;
                    }
                } elseif ($type == "sanaei") {
                    include_once("api/sanaei.php");
                    login($link, $username, $password);
                    $response = addClientToInbound($size, $days, "$link", "$inbound_copy");
                    $create_status = json_decode($response, true);
                    $service_name = $create_status['remark'] ?? $service_name;
                    $service_uuid = $create_status['uuid'] ?? '';
                    $subscribe = updateVlessLink($server['example_link'], $service_uuid, $service_name);
                }

                if (!empty($subscribe)) {
                    $stmt_insert = $connect->prepare("INSERT INTO services (id, uuid, owner, expire, size, name, sub, svname, location, display) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                    $stmt_insert->bind_param("sssiisssss", $service_id, $service_uuid, $user_id, $expire_time, $size, $service_name, $subscribe, $server['name'], $server_display_name, $service_name);
                    $stmt_insert->execute();
                    $stmt_insert->close();

                    $key = json_encode(['inline_keyboard' => [[['text' => "🔎 مشاهده سرویس", 'callback_data' => "service-$service_id"]]]]);
                    $final_subscribe_link = apply_custom_domain($subscribe, $server);
                    $encode_url = urlencode($final_subscribe_link);
                    $caption = "✅ <b>سفارش با کیف پول با موفقیت ثبت شد</b>\n\n" . "👤 <b>کاربر:</b> <code>{$user_id}</code>\n" . "💾 <b>حجم:</b> {$size} گیگ\n" . "🕒 <b>مدت:</b> {$days} روز\n" . "💰 <b>مبلغ:</b> {$amount_fmt} تومان\n" . "📍 <b>سرور:</b> {$server_display_name}\n\n" . "---\n" . "🧩 <b>لینک اشتراک (برای کپی/افزودن):</b>\n" . ($type == 'sanaei' ? "<code>$final_subscribe_link</code>" : $final_subscribe_link);
                    $msg_response = bot('sendPhoto', ['chat_id' => $user_id, 'photo' => "https://api.qrserver.com/v1/create-qr-code/?data=$encode_url&size=800x800", 'caption' => $caption, 'reply_markup' => $key, 'parse_mode'=>'html']);
                    if (isset($msg_response['result']['message_id'])) {
                        $msgid2 = $msg_response['result']['message_id'];
                        $stmt_msg2 = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                        $stmt_msg2->bind_param("is", $msgid2, $user_id);
                        $stmt_msg2->execute();
                        $stmt_msg2->close();
                    }
                    editMessageText($chatid, $messageid, '✅ سفارش با کیف پول پرداخت و ایجاد شد.', null, 'html');
                } else {
                    $stmt_refund2 = $connect->prepare("UPDATE user SET coin = coin + ? WHERE id = ?");
                    $stmt_refund2->bind_param("is", $amount_need, $chatid);
                    $stmt_refund2->execute();
                    $stmt_refund2->close();
                    editMessageText($chatid, $messageid, 'خطا در ایجاد سرویس. مبلغ به کیف پول بازگردانده شد.', null, 'html');
                }
            }
        }
    } elseif(preg_match('/^skip_discount\|(.+)/', $data, $m)){
        $order_code = $m[1];
        // proceed without discount
        $stmt_user = $connect->prepare("UPDATE user SET step='waiting_payment_card', tmp=? WHERE id=?");
        $stmt_user->bind_param("ss", $order_code, $chatid);
        $stmt_user->execute();
        $stmt_user->close();

        $stmt = $connect->prepare("SELECT amount, type, svcode FROM orders WHERE order_code=? LIMIT 1");
        $stmt->bind_param('s', $order_code);
        $stmt->execute();
        $res = $stmt->get_result();
        $ord = $res->fetch_assoc();
        $stmt->close();

        $final_amount = number_format($ord['amount'] ?? 0);
        $back_data = ($ord['type'] ?? '') == 'exten' ? "ex-" . $ord['svcode'] : 'buy';
        $back_key = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => $back_data]]]]);
        // Remove discount prompt and send payment details as a new message (with optional image)
        deleteMessage($chatid, $messageid);
        $sent = send_card_payment_instruction($connect, $chatid, $order_code, $final_amount, $back_key);
        $new_msg_id = $sent['result']['message_id'] ?? null;

        if ($new_msg_id) {
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $new_msg_id, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
    } elseif(preg_match('/^charge_wallet\|(.+)/', $data, $mm)){
        $order_code = $mm[1];
        $stmt_user = $connect->prepare("UPDATE user SET step=?, tmp=? WHERE id=?");
        $step_val = 'await_wallet_amount|' . $order_code;
        $stmt_user->bind_param("sss", $step_val, $order_code, $chatid);
        $stmt_user->execute();
        $stmt_user->close();
        bot('answercallbackquery', ['callback_query_id' => $membercall]);
        editMessageText($chatid, $messageid, "لطفا مبلغ شارژ کیف پول (به تومان) را ارسال کنید:", json_encode(['inline_keyboard' => [[['text' => 'بازگشت', 'callback_data' => 'buy']]]]));
    }elseif($data == 'start_kyc' && $user2['step'] == 'pending_kyc_start'){
        $kyc_message = get_setting($connect, 'kyc_message');
        // Show KYC instructions; replying keyboards are not valid in editMessageText
        editMessageText($chatid, $messageid, $kyc_message, null);
        $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
        $stmt_msg->bind_param("is", $messageid, $chatid);
        $stmt_msg->execute();
        $stmt_msg->close();
        $stmt = $connect->prepare("UPDATE user SET step='waiting_kyc_photo' WHERE id=?");
        $stmt->bind_param("s", $chatid);
        $stmt->execute();
        $stmt->close();
    } elseif($data =="allsrc"){
        deleteMessage($chatid, $user2['msgid']);
        $stmt = $connect->prepare("SELECT * FROM services WHERE owner = ?");
        $stmt->bind_param("s", $chatid);
        $stmt->execute();
        $get = $stmt->get_result();
        if($get->num_rows == 0){ sendMessage($chatid, getMessage($connect, 'error_no_service_found_for_user')); exit(); }
        $keys = []; $i=0;
        while($row = $get->fetch_assoc()){
            $i++; $keys[]= [['text'=>"$i - {$row['display']} - {$row['size']} GB",'callback_data'=>"service-{$row['id']}"]];
        }
        $stmt->close();
        $msgid = sendmessage($chatid, getMessage($connect, 'service_list_prompt'), json_encode(['inline_keyboard'=> $keys]))['result']['message_id'];
        if($msgid) {
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $msgid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
    }
    elseif(preg_match("/^service-/",$data)){
        $service_id = str_replace("service-",null,$data);
        deleteMessage($chatid,$user2['msgid']);

        // --- Fetch service and server details from DB ---
        $stmt = $connect->prepare("SELECT * FROM services WHERE id = ? AND owner = ? LIMIT 1");
        $stmt->bind_param("ss", $service_id, $chatid);
        $stmt->execute();
        $service_res = $stmt->get_result();
        if($service_res->num_rows == 0){
            sendMessage($chatid, getMessage($connect, 'error_service_not_found'));
            exit();
        }
        $service = $service_res->fetch_assoc();
        $stmt->close();

        $stmt_server = $connect->prepare("SELECT * FROM servers WHERE name = ? LIMIT 1");
        $stmt_server->bind_param("s", $service['svname']);
        $stmt_server->execute();
        $server_res = $stmt_server->get_result();
        if($server_res->num_rows == 0){
            sendMessage($chatid, "❌ سرور این سرویس یافت نشد یا در حال حاضر خاموش است.");
            exit();
        }
        $server = $server_res->fetch_assoc();
        $stmt_server->close();

        // --- Initialize variables ---
        $gigabytes = 0;
        $size = round((float)$service['size'], 2);
        $status = "⚠️ نامشخص";
        $panel_status_ok = false;

        // --- Fetch live data from panel ---
        if ($server['type'] == "marzban") {
            $login_response = loginPanel($server['link'], $server['username'], $server['password']);
            $token = $login_response['access_token'] ?? null;
            if ($token) {
                $url = rtrim($server['link'], '/') . "/api/user/{$service['name']}";
                $ch = curl_init($url);
                curl_setopt_array($ch, [
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_TIMEOUT => 10,
                    CURLOPT_CONNECTTIMEOUT => 5,
                    CURLOPT_HTTPHEADER => ['Accept: application/json', 'Authorization: Bearer ' . $token]
                ]);
                if (defined('DISABLE_SSL_VERIFY') && DISABLE_SSL_VERIFY) {
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
                }
                $response = curl_exec($ch);
                curl_close($ch);
                if ($response !== false) {
                    $json = json_decode($response, true);
                    if (isset($json['used_traffic'])) {
                        $gigabytes = round($json['used_traffic'] / 1073741824, 2);
                        $status = ($json['status'] == "active") ? "🟢 فعال" : "🔴 غیرفعال";
                        $panel_status_ok = true;
                    }
                }
            }
        } elseif ($server['type'] == 'sanaei') {
            if (file_exists("api/sanaei.php")) {
                include_once("api/sanaei.php");
                if(function_exists('getClientTraffic')){
                    login($server['link'], $server['username'], $server['password']);
                    $json_response = getClientTraffic($server['link'], $service['name']);
                    $json_data = json_decode($json_response, true);
                    if (isset($json_data['obj'])) {
                        $json = $json_data['obj'];
                        $gigabytes = round(($json['up'] + $json['down']) / 1073741824, 2);
                        $status = ($json['enable'] == true) ? "🟢 فعال" : "🔴 غیرفعال";
                        $panel_status_ok = true;
                    }
                }
            }
        }

        // --- Calculate final status and details ---
        $used_percentage = ($size > 0) ? round(($gigabytes / $size) * 100) : 0;
        include_once('jdf.php');
        $expire_timestamp = (int)$service['expire'];
        $days_remaining = max(0, ceil(($expire_timestamp - time()) / 86400));
        $is_expired = time() > $expire_timestamp;
        
        // Final status override if expired or traffic limit reached
        if ($is_expired || ($panel_status_ok && $used_percentage >= 100)) {
            $status = "🔴 غیرفعال";
        }

        // --- Prepare UI components ---
        $key_array = [
            [['text'=>'📇 تغییر نام','callback_data'=>"chn-$service_id"],['text'=>'🔗 لینک اتصال','callback_data'=>"inf-$service_id"]],
            [['text'=>"🔋 تمدید سرویس","callback_data"=>"ex-$service_id"]],
            [['text'=>"🔙 بازگشت به لیست",'callback_data'=>"allsrc"]]
        ];
        if ($status == "🔴 غیرفعال") {
            $key_array[0][0]['text'] = '📇 تغییر نام (غیرفعال)';
            $key_array[0][0]['callback_data'] = 'no';
            $key_array[0][1]['text'] = '🔗 لینک اتصال (غیرفعال)';
            $key_array[0][1]['callback_data'] = 'no';
        }

        $progressBar = create_progress_bar($used_percentage, 10);
        $used_gb_formatted = number_format($gigabytes, 2);
        $total_gb_formatted = number_format($size, 2);
        $expire_date_jalali = jdate('Y/m/d', $expire_timestamp);

        // --- Build the final message caption using HTML ---
        $caption = "⚙️ <b>مدیریت سرویس</b>\n";
        $caption .= "➖➖➖➖➖➖➖➖\n";
        $caption .= "🏷️ <b>نام:</b> <code>{$service['display']}</code>\n";
        $caption .= "📡 <b>سرور:</b> {$service['location']}\n\n";
        $caption .= "📊 <b>وضعیت ترافیک:</b>\n";
        $caption .= "<code>{$progressBar}</code>\n";
        $caption .= "<i>مصرف:</i> <code>{$used_gb_formatted}</code> / <code>{$total_gb_formatted}</code> گیگابایت ({$used_percentage}%)\n\n";
        $caption .= "🗓️ <b>اعتبار زمانی:</b>\n";
        $caption .= "<i>باقیمانده:</i> <b>{$days_remaining}</b> روز\n";
        $caption .= "<i>تاریخ انقضا:</i> <code>{$expire_date_jalali}</code>\n";
        $caption .= "➖➖➖➖➖➖➖➖\n";
        $caption .= "🚦 <b>وضعیت کلی:</b> {$status}";

        if (!$panel_status_ok && !$is_expired) {
            $caption .= "\n\n⚠️ <i>خطا در دریافت اطلاعات لحظه‌ای از پنل.</i>";
        }

        // --- Generate and send banner image (if enabled) ---
        $send_img_setting = get_setting($connect, 'service_image_enabled');
        $should_send_img = ($send_img_setting === null || $send_img_setting === '' || $send_img_setting === 'on');
        if ($should_send_img) {
            // Prefer uploaded template from settings if available
            $template_path = null;
            $uploaded_rel = get_setting($connect, 'service_template_path');
            if (!empty($uploaded_rel)) {
                $maybe = __DIR__ . '/' . ltrim($uploaded_rel, '/');
                if (is_readable($maybe)) {
                    $template_path = $maybe;
                }
            }
            if (!$template_path) {
                $template_path = __DIR__ . '/assets/service_template.jpg';
                if (!is_readable($template_path)) {
                    // allow png too (in case you put the provided template as png)
                    $alt = __DIR__ . '/assets/service_template.png';
                    if (is_readable($alt)) $template_path = $alt; else $template_path = null;
                }
            }
            // Use English text on image to ensure fonts render correctly
            $lines_on_card = build_service_card_text_en($service, $server['display_name'], $gigabytes, $size, $days_remaining);
            $out_path = __DIR__ . '/data/service_card_' . $service_id . '_' . time() . '.jpg';
            $ok_img = generate_service_card_image($template_path, $out_path, $lines_on_card, [ 'line_spacing' => 1.85 ]);
            $photo_to_send = $out_path;
            if (!$ok_img || !is_file($out_path)) {
                // fallback to template or dynamic image
                $photo_to_send = $template_path;
                if (!$photo_to_send || !is_readable($photo_to_send)) {
                    // generate a default template and use it
                    $tmp_fallback = ih_default_template();
                    $photo_to_send = $tmp_fallback;
                }
            }
            // Send photo
            $response = bot('sendPhoto', [
                'chat_id' => $chatid,
                'photo' => new CURLFile(realpath($photo_to_send)),
                'caption' => $caption,
                'parse_mode' => 'html',
                'reply_markup' => json_encode(['inline_keyboard' => $key_array])
            ]);
            if (isset($out_path) && is_file($out_path)) @unlink($out_path);
            $msgid = $response['result']['message_id'] ?? null;
        } else {
            // Send only text without image
            $response = bot('sendMessage', [
                'chat_id' => $chatid,
                'text' => $caption,
                'parse_mode' => 'html',
                'reply_markup' => json_encode(['inline_keyboard' => $key_array])
            ]);
            $msgid = $response['result']['message_id'] ?? null;
        }
        if($msgid) {
            $stmt_msg = $connect->prepare("UPDATE user SET step='none', msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $msgid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
    }
    elseif(preg_match('/^inf-/',$data)){
        $service_id = str_replace("inf-",null,$data);
        deleteMessage($chatid,$user2['msgid']);
        $stmt = $connect->prepare("SELECT sub, display, svname FROM services WHERE id = ? LIMIT 1");
        $stmt->bind_param("s", $service_id);
        $stmt->execute();
        $service_res = $stmt->get_result();
        $service = $service_res->fetch_assoc();
        $stmt->close();

        // Retrieve server to apply custom domain if available
        $server = [];
        if (!empty($service['svname'])) {
            $stmt_server = $connect->prepare("SELECT custom_domain FROM servers WHERE name = ? LIMIT 1");
            $stmt_server->bind_param("s", $service['svname']);
            $stmt_server->execute();
            $server = $stmt_server->get_result()->fetch_assoc() ?: [];
            $stmt_server->close();
        }

        $key = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت به مدیریت', 'callback_data' => "service-$service_id"]]]]);
        $sub = apply_custom_domain($service['sub'], $server);
        $caption = "🔗 <b>لینک اتصال سرویس</b>\n\n" . "🏷 <b>نام سرویس:</b> <code>{$service['display']}</code>\n\n" . "<code>$sub</code>\n\n" . "<i>این لینک محرمانه است، آن را در اختیار دیگران قرار ندهید.</i>";
        $msgid = sendMessage($chatid, $caption, $key, 'html')['result']['message_id'];
        if($msgid) {
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $msgid, $chatid);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
    }
    elseif(preg_match('/^chn-(\d+)/',$data, $matches)){
        $service_id = $matches[1];
        $stmt = $connect->prepare("SELECT display, last_name_change_at FROM services WHERE id = ? AND owner = ? LIMIT 1");
        $stmt->bind_param("ss", $service_id, $chatid);
        $stmt->execute();
        $service_res = $stmt->get_result();
        if ($service_res->num_rows == 0) { bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "❌ سرویس یافت نشد."]); exit(); }
        $service = $service_res->fetch_assoc();
        $stmt->close();
        $cooldown = 600; $last_change = $service['last_name_change_at'];
        if ($last_change !== null && (time() - $last_change) < $cooldown) {
            include_once('jdf.php');
            bot('answercallbackquery', ['callback_query_id' => $membercall, 'text' => "⏳ شما در ساعت " . jdate('H:i:s', $last_change + $cooldown) . " می‌توانید دوباره تلاش کنید.", 'show_alert' => true]);
            exit();
        }
        deleteMessage($chatid,$user2['msgid']);
        $key=json_encode(['inline_keyboard'=>[[['text'=>'🔙 بازگشت','callback_data'=>"service-$service_id"]]]]);
        $msgid = sendmessage($chatid, getMessage($connect, 'ask_for_new_service_name', ['service_name' => $service['display']]), $key)['result']['message_id'];
        $stmt_user = $connect->prepare("UPDATE user SET step=?,msgid=? WHERE id=?");
        $step = "change-$service_id";
        $stmt_user->bind_param("sis", $step, $msgid, $chatid);
        $stmt_user->execute();
        $stmt_user->close();
    }
    
    elseif(preg_match('/^ex-/',$data)){
        $id = str_replace('ex-',null,$data);
        deleteMessage($chatid,$user2['msgid']);
        $stmt_service = $connect->prepare("SELECT display, svname FROM services WHERE id = ? LIMIT 1");
        $stmt_service->bind_param("s", $id);
        $stmt_service->execute();
        $service = $stmt_service->get_result()->fetch_assoc();
        $stmt_service->close();
        
        $stmt_server = $connect->prepare("SELECT id FROM servers WHERE name = ? LIMIT 1");
        $stmt_server->bind_param("s", $service['svname']);
        $stmt_server->execute();
        $sv = $stmt_server->get_result()->fetch_assoc();
        $stmt_server->close();
        
        $key=json_encode(['inline_keyboard'=>[[['text'=>'لوکیشن سرویس فعلی','callback_data'=>"selectserver|{$sv['id']}"]]]]);
        $msgid = sendmessage($chatid, getMessage($connect, 'extend_service_welcome', ['service_name' => $service['display']]), $key)['result']['message_id'];
        
        $stmt_user = $connect->prepare("UPDATE user SET step='exten',msgid=?,want=? WHERE id=?");
        $stmt_user->bind_param("iss", $msgid, $id, $chatid);
        $stmt_user->execute();
        $stmt_user->close();
    }
    exit();
} 
elseif (isset($text) || isset($photoid)) {
    if (is_dir("block/$chat_id")) { exit(); }

    // --- اطمینان از داشتن وضعیت مرحله کاربر ---
    if (!isset($user) || !is_array($user)) {
        $stmt_tmp = $connect->prepare("SELECT step, msgid FROM user WHERE id = ? LIMIT 1");
        $stmt_tmp->bind_param("s", $chat_id);
        $stmt_tmp->execute();
        $res_tmp = $stmt_tmp->get_result();
        $user = $res_tmp->num_rows ? $res_tmp->fetch_assoc() : ['step' => 'none'];
        $stmt_tmp->close();
    }

    // --- مرحله‌هایی که باید از گیت عضویت عبور کنند (ارسال فیش/احراز هویت) ---
    $current_step = $user['step'] ?? 'none';
    $skip_join_for_steps = ['waiting_payment_card', 'waiting_payment_crypto', 'waiting_payment_stars', 'waiting_wallet_card', 'waiting_wallet_crypto', 'waiting_kyc_photo', 'awaiting_discount_code'];
    if (is_string($current_step) && strpos($current_step, 'await_wallet_amount') === 0) {
        $skip_join_for_steps[] = $current_step;
    }

    // --- گیت عضویت با استثنا برای آپلود فیش و KYC ---
    // Allow photo messages (receipts) to bypass forced-join gate so تازه‌واردها نیز بتوانند رسید بفرستند
    if ($join_status === 'on' && $chat_id != $admin_id && !isset($photoid) && !in_array($current_step, $skip_join_for_steps, true)) {
        $join_result = joinChannels($connect, $chat_id);
        if ($join_result !== true) {
            $buttons = [];
            foreach ($join_result as $channel) {
                $buttons[] = [[
                    'text' => "📪 عضویت در " . str_replace("@", "", $channel),
                    'url'  => "https://t.me/" . str_replace("@", "", $channel)
                ]];
            }
            $buttons[] = [['text' => '✅ عضو شدم', 'callback_data' => 'joined']];
            sendMessage(
                $chat_id,
                "⚠️ کاربر گرامی؛ برای استفاده از ربات لازم است در کانال(های) زیر عضو شوید.",
                json_encode(['inline_keyboard' => $buttons])
            );
            exit();
        }
    }
    
    // Wallet menu via text message (any text containing "کیف پول")
    if (isset($text) && strpos($text, 'کیف پول') !== false) {
        $stmtb = $connect->prepare("SELECT coin FROM user WHERE id = ? LIMIT 1");
        $stmtb->bind_param("s", $chat_id);
        $stmtb->execute();
        $rowb = $stmtb->get_result()->fetch_assoc();
        $stmtb->close();
        $balance_text = number_format((int)($rowb['coin'] ?? 0));
        $msg = "👛 <b>کیف پول شما</b>\n\n💰 موجودی: <b>{$balance_text}</b> تومان";
        $kb = json_encode(['inline_keyboard' => [
            [ ['text' => '➕ شارژ کارت به کارت', 'callback_data' => 'wallet_charge_card'], ['text' => '➕ شارژ رمز ارز', 'callback_data' => 'wallet_charge_crypto'] ],
            [ ['text' => '🔙 بازگشت', 'callback_data' => 'buy'] ]
        ]]);
        $sent = sendMessage($chat_id, $msg, $kb, 'html');
        $msgid_wallet = $sent['result']['message_id'] ?? null;
        if ($msgid_wallet) {
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $msgid_wallet, $chat_id);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
        exit();
    }
    
    // Wallet top-up amount entry (online gateway used elsewhere)
    if (isset($text) && preg_match('/^await_wallet_amount\|(.+)/', $user['step'], $m2)) {
        $order_code = $m2[1];
        $amount = (int)preg_replace('/[^0-9]/', '', (string)$text);
        if ($amount <= 0) {
            sendMessage($chat_id, 'لطفا تنها مبلغ معتبر (عدد) را ارسال کنید.');
            exit();
        }
        $base = PAYMENT_CALLBACK_BASE ?: ((isset($_SERVER['REQUEST_SCHEME']) ? $_SERVER['REQUEST_SCHEME'] : 'https') . '://' . $_SERVER['HTTP_HOST']);
        $pay_url = rtrim($base, '/') . '/api/payz.php?amount=' . $amount . '&chat=' . $chat_id;
        $kb = json_encode(['inline_keyboard' => [[['text' => 'پرداخت آنلاین', 'url' => $pay_url]], [['text' => 'بازگشت', 'callback_data' => 'buy']]]]);
        $msg = "برای شارژ کیف پول به مبلغ <b>" . number_format($amount) . "</b> تومان، روی دکمه زیر بزنید و پرداخت را انجام دهید.\nپس از موفقیت پرداخت، موجودی شما به صورت خودکار افزایش می‌یابد.";
        sendMessage($chat_id, $msg, $kb, 'html');
        $stmt_user = $connect->prepare("UPDATE user SET step='none', tmp=NULL WHERE id=?");
        $stmt_user->bind_param("s", $chat_id);
        $stmt_user->execute();
        $stmt_user->close();
        exit();
    }

    // Wallet receipts (manual): card or crypto
    if ((($user['step'] ?? '') === 'waiting_wallet_card' || ($user['step'] ?? '') === 'waiting_wallet_crypto') && isset($photoid)) {
        $code = (string)($user['tmp'] ?? '');
        if ($code === '') { sendMessage($chat_id, '⛔️ خطا: کد پیگیری یافت نشد.'); exit(); }
        $connect->query("CREATE TABLE IF NOT EXISTS wallet_topups (id INT AUTO_INCREMENT PRIMARY KEY, code VARCHAR(32) UNIQUE, chat_id BIGINT, amount INT, method ENUM('card','crypto') NOT NULL, status ENUM('pending','approved','rejected') DEFAULT 'pending', created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP)");
        $stmts = $connect->prepare("SELECT amount, method, status FROM wallet_topups WHERE code=? AND chat_id=? LIMIT 1");
        $stmts->bind_param('ss', $code, $chat_id);
        $stmts->execute();
        $res = $stmts->get_result();
        $row = $res->fetch_assoc();
        $stmts->close();
        if (!$row) { sendMessage($chat_id, '⛔️ درخواست شارژ شما یافت نشد. مجدد تلاش کنید.'); exit(); }
        if ($row['status'] !== 'pending') { sendMessage($chat_id, '⛔️ این درخواست قبلاً پردازش شده است.'); exit(); }
        $amount = number_format((int)$row['amount']);
        $method_text = ($row['method'] === 'card') ? 'کارت به کارت' : 'رمز ارز';
        $username = isset($from_id['username']) ? " (@{$from_id['username']})" : "";
        $admin_caption = "🧾 <b>درخواست شارژ کیف پول ({$method_text})</b>\n\n" .
                         "👤 <b>کاربر:</b> <code>{$chat_id}</code>{$username}\n" .
                         "💰 <b>مبلغ:</b> {$amount} تومان\n" .
                         "🔑 <b>کد پیگیری:</b> <code>{$code}</code>\n\n" .
                         "تایید شود؟";
        $kb_admin = json_encode(['inline_keyboard' => [
            [ ['text' => '✅', 'callback_data' => "approve_wallet|{$code}"], ['text' => '❌', 'callback_data' => "reject_wallet|{$code}"] ]
        ]]);
        bot('sendPhoto', [ 'chat_id' => $admin_id, 'photo' => $photoid, 'caption' => $admin_caption, 'parse_mode' => 'html', 'reply_markup' => $kb_admin ]);
        sendMessage($chat_id, '✅ رسید شما دریافت شد و در انتظار تایید مدیریت است.', $start);
        $stmt_user = $connect->prepare("UPDATE user SET step='none', tmp=NULL WHERE id=?");
        $stmt_user->bind_param('s', $chat_id);
        $stmt_user->execute();
        $stmt_user->close();
        exit();
    }

    // توجه: از ارسال عمومی هر عکس به ادمین صرف‌نظر شد. رسید فقط در مراحل مربوطه پذیرفته می‌شود.

    // Handle discount code input (user typed a code while awaiting_discount_code)
    if ($user['step'] == 'awaiting_discount_code' && isset($text) && trim($text) !== '') {
        $order_code = $user['tmp'];
        $msg_id = $user['msgid'];
        $clean_text = trim($text);
        // allow users to type "بدون کد" or "skip" to proceed without a discount
        if ($clean_text === 'بدون کد' || $clean_text === 'بدونکد' || strtolower($clean_text) === 'skip') {
            $stmt_user = $connect->prepare("UPDATE user SET step='waiting_payment_card', tmp=? WHERE id=?");
            $stmt_user->bind_param("ss", $order_code, $chat_id);
            $stmt_user->execute();
            $stmt_user->close();

            $stmt = $connect->prepare("SELECT amount, type, svcode FROM orders WHERE order_code=? LIMIT 1");
            $stmt->bind_param('s', $order_code);
            $stmt->execute();
            $res = $stmt->get_result();
            $ord = $res->fetch_assoc();
            $stmt->close();

            $final_amount = number_format($ord['amount'] ?? 0);
            $back_data = ($ord['type'] ?? '') == 'exten' ? "ex-" . $ord['svcode'] : 'buy';
            $back_key = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => $back_data]]]]);
            // Remove discount prompt and send payment details as a new message (with optional image)
            deleteMessage($chat_id, $msg_id);
            $sent = send_card_payment_instruction($connect, $chat_id, $order_code, $final_amount, $back_key);
            $new_msg_id = $sent['result']['message_id'] ?? null;

            if ($new_msg_id) {
                $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                $stmt_msg->bind_param("is", $new_msg_id, $chat_id);
                $stmt_msg->execute();
                $stmt_msg->close();
            }
        } else {
            $result = apply_discount_code($connect, $order_code, (string)$chat_id, $clean_text);
            if ($result['success']) {
                $stmt_user = $connect->prepare("UPDATE user SET step='waiting_payment_card', tmp=? WHERE id=?");
                $stmt_user->bind_param("ss", $order_code, $chat_id);
                $stmt_user->execute();
                $stmt_user->close();

                $stmt = $connect->prepare("SELECT amount, type, svcode FROM orders WHERE order_code=? LIMIT 1");
                $stmt->bind_param('s', $order_code);
                $stmt->execute();
                $res = $stmt->get_result();
                $ord = $res->fetch_assoc();
                $stmt->close();

                $final_amount = number_format($ord['amount'] ?? $result['final_amount']);
                $back_data = ($ord['type'] ?? '') == 'exten' ? "ex-" . $ord['svcode'] : 'buy';
                $back_key = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => $back_data]]]]);
                // Send a new message instead of editing the prompt (optionally as image)
                deleteMessage($chat_id, $msg_id);
                $prefix_html = '✅ ' . $result['message'];
                $sent = send_card_payment_instruction($connect, $chat_id, $order_code, $final_amount, $back_key, $prefix_html);
                $new_msg_id = $sent['result']['message_id'] ?? null;
                if ($new_msg_id) {
                    $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                    $stmt_msg->bind_param("is", $new_msg_id, $chat_id);
                    $stmt_msg->execute();
                    $stmt_msg->close();
                }
            } else {
                $skip_key = json_encode(['inline_keyboard' => [[['text' => 'بدون کد', 'callback_data' => "skip_discount|$order_code"]]]]);
                // Send a new error message; do not edit the prompt
                deleteMessage($chat_id, $msg_id);
                $sent = sendMessage($chat_id, "❌ {$result['message']}\n\nکد دیگری ارسال کنید یا بدون کد ادامه دهید.", $skip_key, 'html');
                $new_msg_id = $sent['result']['message_id'] ?? null;
                if ($new_msg_id) {
                    $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                    $stmt_msg->bind_param("is", $new_msg_id, $chat_id);
                    $stmt_msg->execute();
                    $stmt_msg->close();
                }
            }
        }
        exit();
    }
    
    $is_start_command = false;
    $start_payload = null;
    if (is_string($text) && strpos($text, '/start') === 0) {
        $is_start_command = true;
        $parts = explode(' ', $text, 2);
        if (isset($parts[1])) {
            $start_payload = trim($parts[1]);
        }
    }

    if ($is_start_command || $text == '🔙 منو اصلی') {
        $attached_referrer_id = null;
        if ($is_start_command && empty($user) && !empty($start_payload) && !empty($chat_id) && function_exists('referral_resolve_referrer')) {
            $resolved_referrer = referral_resolve_referrer($connect, $start_payload, (string)$chat_id);
            if ($resolved_referrer && function_exists('referral_assign_referred_by')) {
                if (referral_assign_referred_by($connect, (string)$chat_id, $resolved_referrer)) {
                    $attached_referrer_id = $resolved_referrer;
                }
            }
        }
        if (!empty($chat_id) && function_exists('referral_ensure_user_code')) {
            referral_ensure_user_code($connect, (string)$chat_id);
        }
        $stmt = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
        $stmt->bind_param("s", $chat_id);
        $stmt->execute();
        $stmt->close();
        sendmessage($chat_id, getMessage($connect, 'welcome_message'), $start);
        if (!empty($attached_referrer_id)) {
            $ref_text = getMessage($connect, 'referral_welcome_referred', ['referrer_id' => $attached_referrer_id]);
            if ($ref_text) {
                sendMessage($chat_id, $ref_text, $start, 'html');
            }
        }
        exit();
    }
    
    if ($chat_id == $admin_id) {
        if (preg_match('/^set_price\|(\d+)\|(\d+)/', $user['step'], $matches)) {
            $server_id = $matches[1];
            $plan_id = $matches[2];
            $price = $text;

            if (is_numeric($price)) {
                $step = "set_star_price|{$server_id}|{$plan_id}|{$price}";
                $stmt_user = $connect->prepare("UPDATE user SET step=? WHERE id=?");
                $stmt_user->bind_param("ss", $step, $chat_id);
                $stmt_user->execute();
                $stmt_user->close();

                deleteMessage($chat_id, $message_id);
                editMessageText($chat_id, $user['msgid'], "اکنون تعداد استارز را وارد کنید.\n\n(برای حذف پرداخت با استارز، عدد <code>0</code> را ارسال کنید)", null, 'html');
            } else {
                sendMessage($chat_id, "❌ لطفاً فقط عدد وارد کنید.");
            }
        } elseif (preg_match('/^set_star_price\|(\d+)\|(\d+)\|(\d+)/', $user['step'], $m)) {
            $server_id = $m[1];
            $plan_id = $m[2];
            $price = (int)$m[3];
            $stars = $text;

            if (is_numeric($stars)) {
                $stars = (int)$stars;
                if ($price > 0 || $stars > 0) {
                    $stmt = $connect->prepare("INSERT INTO plan_prices (plan_id, server_id, price, stars_price) VALUES (?, ?, ?, ?) ON DUPLICATE KEY UPDATE price = VALUES(price), stars_price = VALUES(stars_price)");
                    $stmt->bind_param("iiii", $plan_id, $server_id, $price, $stars);
                } else {
                    $stmt = $connect->prepare("DELETE FROM plan_prices WHERE plan_id = ? AND server_id = ?");
                    $stmt->bind_param("ii", $plan_id, $server_id);
                }
                $stmt->execute();
                $stmt->close();

                $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
                $stmt_user->bind_param("s", $chat_id);
                $stmt_user->execute();
                $stmt_user->close();

                deleteMessage($chat_id, $message_id);
                deleteMessage($chat_id, $user['msgid']);
                sendMessage($chat_id, "✅ قیمت با موفقیت ثبت شد.", $panel);
            } else {
                sendMessage($chat_id, "❌ لطفاً فقط عدد وارد کنید.");
            }
        }
        elseif ($text == "/panel" || $text == "بازگشت به پنل" || $text == "❌ لغو عملیات" || $text == "❌ لغو") {
            $stmt = $connect->prepare("UPDATE user SET step='none', tmp=NULL WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendmessage($chat_id, "به پنل مدیریت خوش آمدید.", $panel);
        }
        elseif ($text == 'آمار' || $text == '/analytics' || $text == '📊 گزارش سریع') {
            $stats = compute_extended_stats($connect);
            $lines = [];
            $lines[] = "📊 <b>گزارش مدیریتی امروز</b>";
            $lines[] = "👥 کاربران فعال: <code>" . format_number_plain($stats['active_users']) . "</code>";
            $lines[] = "🆕 کاربران ۷ روز اخیر: <code>" . format_number_plain($stats['new_users_7_days']) . "</code>";
            $lines[] = "⏳ سرویس‌های رو به اتمام (۷ روز آینده): <code>" . format_number_plain($stats['expiring_week']) . "</code>";
            $lines[] = "🧾 سفارشات در انتظار تایید: <code>" . format_number_plain($stats['pending_orders']) . "</code>";
            $lines[] = "💰 درآمد ۳۰ روز اخیر: <code>" . format_number_plain($stats['revenue_30_days']) . "</code> تومان";
            if ($stats['revenue_growth'] !== null) {
                $growth = (float)$stats['revenue_growth'];
                $trendEmoji = $growth >= 0 ? '🔼' : '🔻';
                $lines[] = $trendEmoji . ' رشد نسبت به ماه قبل: <code>' . format_number_plain(abs($growth)) . '</code>%';
            }
            $lines[] = "💳 میانگین مبلغ سفارش تایید شده: <code>" . format_number_plain(round($stats['avg_order_amount'])) . "</code> تومان";

            if (!empty($stats['payment_breakdown'])) {
                $lines[] = "\n💳 <b>تفکیک روش‌های پرداخت</b>";
                foreach ($stats['payment_breakdown'] as $item) {
                    $label = payment_type_label($item['type']);
                    $lines[] = "• {$label}: <code>" . format_number_plain($item['orders_count']) . "</code> سفارش – <code>" . format_number_plain($item['total_amount']) . "</code> تومان";
                }
            }

            if (!empty($stats['retention'])) {
                $retention = $stats['retention'];
                $lines[] = "\n🔁 <b>رفتار مشتریان {$retention['period_days']} روز اخیر</b>";
                $lines[] = "• کاربران فعال: <code>" . format_number_plain($retention['active_customers']) . "</code> | بازگشتی: <code>" . format_number_plain($retention['returning_customers']) . "</code> | جدید: <code>" . format_number_plain($retention['new_customers']) . "</code>";
                $lines[] = "• نرخ بازگشت: <code>" . format_decimal_plain($retention['repeat_rate'], 1) . "</code>% | میانگین سفارش برای هر کاربر: <code>" . format_decimal_plain($retention['average_orders_per_customer'], 2) . "</code>";
                if (!empty($retention['dormant_customers'])) {
                    $lines[] = "• کاربران غیرفعال: <code>" . format_number_plain($retention['dormant_customers']) . "</code>";
                }
            }

            $health = $stats['server_health'];
            $lines[] = "\n🛰 <b>وضعیت سرورها</b>";
            $lines[] = "• آنلاین: <code>" . format_number_plain($health['online']) . "</code> | نگهداری: <code>" . format_number_plain($health['maintenance']) . "</code> | خاموش: <code>" . format_number_plain($health['offline']) . "</code>";
            if ($health['total_capacity'] > 0) {
                $capacityPercent = round(($health['total_load'] / max(1, $health['total_capacity'])) * 100);
                $lines[] = "• استفاده کلی ظرفیت: <code>" . format_number_plain($health['total_load']) . "</code> از <code>" . format_number_plain($health['total_capacity']) . "</code> (" . $capacityPercent . "%)";
            }
            if (!empty($stats['region_breakdown'])) {
                $lines[] = "• مناطق پربازده:";
                $regions = array_slice($stats['region_breakdown'], 0, 3);
                foreach ($regions as $region) {
                    $lines[] = '  ◽️ ' . escape_html($region['region']) . ' – سرور: <code>' . format_number_plain($region['server_count']) . '</code> | استفاده: <code>' . ($region['utilization'] !== null ? format_number_plain($region['utilization']) . '%': '---') . '</code>';
                }
            }
            if (!empty($stats['top_servers'])) {
                $lines[] = "• برترین سرورها:";
                foreach (array_slice($stats['top_servers'], 0, 3) as $top_server) {
                    $server_name = escape_html($top_server['display_name'] ?? ('سرور #' . $top_server['id']));
                    $util_text = ($top_server['utilization'] !== null) ? format_number_plain($top_server['utilization']) . '%' : '---';
                    $lines[] = "  ◽️ {$server_name} – سرویس فعال: <code>" . format_number_plain($top_server['active_services']) . "</code> | استفاده: <code>{$util_text}</code>";
                }
            }
            if (!empty($stats['daily_trend'])) {
                $latest = end($stats['daily_trend']);
                if ($latest) {
                    $lines[] = "\n📅 امروز: <code>" . format_number_plain($latest['orders_count']) . "</code> سفارش – <code>" . format_number_plain($latest['total_amount']) . "</code> تومان";
                }
                reset($stats['daily_trend']);
            }

            sendMessage($chat_id, implode("\n", $lines), $panel, 'html');
        }
        elseif ($text == '/servers' || $text == '🛰 وضعیت سرورها') {
            $servers = fetch_servers_with_metadata($connect);
            if (empty($servers)) {
                sendMessage($chat_id, '⚠️ هنوز سروری در سامانه ثبت نشده است.', $panel, 'html');
            } else {
                $online = $offline = $maintenance = 0;
                $blocks = [];
                foreach ($servers as $server) {
                    $status = $server['health_status'] ?? '';
                    if ($status === 'maintenance') {
                        $maintenance++;
                    } elseif ($status === 'online' || $status === 'warning') {
                        $online++;
                    } else {
                        $offline++;
                    }
                    switch ($status) {
                        case 'maintenance':
                            $emoji = '🛠';
                            break;
                        case 'warning':
                            $emoji = '⚠️';
                            break;
                        case 'online':
                            $emoji = '🟢';
                            break;
                        default:
                            $emoji = '🔴';
                            break;
                    }
                    $region = !empty($server['region']) ? escape_html($server['region']) : 'نامشخص';
                    $name = escape_html($server['display_name'] ?? $server['name']);
                    $util_text = ($server['utilization'] !== null) ? format_number_plain($server['utilization']) . '%' : '---';
                    $capacity_text = ($server['max_capacity'] > 0)
                        ? format_number_plain($server['current_load']) . ' / ' . format_number_plain($server['max_capacity'])
                        : format_number_plain($server['current_load']);
                    $tag_strings = [];
                    if (!empty($server['tags'])) {
                        foreach ($server['tags'] as $tag) {
                            $safe_tag = '#' . preg_replace('/\s+/', '_', $tag['name']);
                            $tag_strings[] = escape_html($safe_tag);
                        }
                    }
                    $tags_text = !empty($tag_strings) ? implode(' ', $tag_strings) : '—';
                    $note = '';
                    if (!empty($server['maintenance_note'])) {
                        $note = "\n└ <i>" . escape_html($server['maintenance_note']) . '</i>';
                    }
                    $blocks[] = sprintf(
                        "%s <b>%s</b> (%s)\n├ بار: <code>%s</code> | استفاده: <code>%s</code>\n├ وضعیت: %s%s\n└ برچسب‌ها: %s%s",
                        $emoji,
                        $name,
                        $region,
                        $capacity_text,
                        $util_text,
                        ($server['status'] === 'on' ? 'فعال' : 'غیرفعال'),
                        ((int)$server['maintenance_mode'] === 1 ? ' | نگهداری' : ''),
                        $tags_text,
                        $note
                    );
                }
                $summary = "🛰 <b>وضعیت لحظه‌ای سرورها</b>\nآنلاین: <code>" . format_number_plain($online) . "</code> | نگهداری: <code>" . format_number_plain($maintenance) . "</code> | خاموش: <code>" . format_number_plain($offline) . "</code>";
                $message = $summary . "\n\n" . implode("\n\n", $blocks);
                sendMessage($chat_id, $message, $panel, 'html');
            }
        }
        elseif ($text == "پیام همگانی") {
            $stmt = $connect->prepare("UPDATE user SET step = 's2a' WHERE id = ?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendmessage($chat_id, "لطفا متن مورد نظر برای ارسال همگانی را بفرستید.", $backp);
        }
        elseif ($user['step'] == 's2a') {
            $stmt_user = $connect->prepare("UPDATE user SET step = 'none' WHERE id = ?");
            $stmt_user->bind_param("s", $chat_id);
            $stmt_user->execute();
            $stmt_user->close();
            
            $encoded_text = urlencode($text);
            $photo_id_val = $photoid ?? '';
            
            $stmt_send = $connect->prepare("INSERT INTO `send`(`step`, `text`, `from`, `user`, `photo`) VALUES ('send', ?, ?, '0', ?)");
            $stmt_send->bind_param("sss", $encoded_text, $chat_id, $photo_id_val);
            $stmt_send->execute();
            $stmt_send->close();
            
            sendmessage($chat_id, "✅ پیام شما در صف ارسال برای تمام کاربران قرار گرفت.", $panel);
        }
        elseif ($text == "افزایش موجودی") {
            $stmt = $connect->prepare("UPDATE user SET step='afz-id' WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendmessage($chat_id, "ایدی عددی کاربر را وارد کنید:", $backp);
        }
        elseif ($user['step'] == "afz-id") {
            $step = "am-$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("ss", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendmessage($chat_id, "مبلغ افزایش موجودی را به تومان وارد کنید:", $backp);
        }
        elseif (preg_match("/^am-/", $user['step'])) {
            $target_id = str_replace("am-", "", $user['step']);
            $stmt = $connect->prepare("UPDATE user SET coin=coin+? WHERE id=?");
            $stmt->bind_param("is", $text, $target_id);
            $stmt->execute();
            $stmt->close();
            
            $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
            $stmt_user->bind_param("s", $chat_id);
            $stmt_user->execute();
            $stmt_user->close();
            
            sendmessage($target_id, "✅ مبلغ " . number_format($text) . " تومان به حساب شما توسط مدیریت اضافه شد.");
            sendmessage($chat_id, "✅ موجودی کاربر با موفقیت افزایش یافت.", $panel);
        }
        elseif ($text == "کاهش موجودی") {
            $stmt = $connect->prepare("UPDATE user SET step='kah-id' WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendmessage($chat_id,"ایدی عددی کاربر؟",$backp);
        }
        elseif ($user['step'] == "kah-id") {
            $step = "mak-$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("ss", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendmessage($chat_id,"چند هزار تومان؟",$backp);
        }
        elseif (preg_match("/^mak\-(.*?)/", $user['step'])) {
            $cc = str_replace("mak-", null, $user['step']);
            $stmt = $connect->prepare("UPDATE user SET coin=coin-? WHERE id=?");
            $stmt->bind_param("is", $text, $cc);
            $stmt->execute();
            $stmt->close();
            
            $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
            $stmt_user->bind_param("s", $chat_id);
            $stmt_user->execute();
            $stmt_user->close();
            sendmessage($chat_id, "انجام شد.", $panel);
        }
        elseif ($text == "دریافت اطلاعات کاربر") {
            $stmt = $connect->prepare("UPDATE user SET step='getinfo' WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendmessage($chat_id,"ایدی عددی، یوزرنیم (@username) یا UUID سرویس کاربر را وارد کنید:",$backp);
        }
        elseif ($user['step'] == "getinfo") {
            $stmt = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            
            $identifier = trim($text);
            $user_info = null;
            $stmt_user = null;

            if (is_numeric($identifier)) {
                $stmt_user = $connect->prepare("SELECT * FROM user WHERE id = ? LIMIT 1");
                $stmt_user->bind_param("s", $identifier);
            } elseif (strpos($identifier, '@') === 0) {
                $username = substr($identifier, 1);
                $stmt_user = $connect->prepare("SELECT * FROM user WHERE username = ? LIMIT 1");
                $stmt_user->bind_param("s", $username);
            } else {
                $stmt_owner = $connect->prepare("SELECT owner FROM services WHERE uuid = ? OR name = ? LIMIT 1");
                $stmt_owner->bind_param("ss", $identifier, $identifier);
                $stmt_owner->execute();
                $owner_res = $stmt_owner->get_result();
                if ($owner_res->num_rows > 0) {
                    $owner_id = $owner_res->fetch_assoc()['owner'];
                    $stmt_user = $connect->prepare("SELECT * FROM user WHERE id = ? LIMIT 1");
                    $stmt_user->bind_param("s", $owner_id);
                }
                $stmt_owner->close();
            }

            if ($stmt_user) {
                $stmt_user->execute();
                $user_res = $stmt_user->get_result();
                if ($user_res->num_rows > 0) {
                    $user_info = $user_res->fetch_assoc();
                }
                $stmt_user->close();
            }

            if ($user_info) {
                $user_id = $user_info['id'];
                $stmt_services = $connect->prepare("SELECT * FROM services WHERE owner = ?");
                $stmt_services->bind_param("s", $user_id);
                $stmt_services->execute();
                $services_res = $stmt_services->get_result();
                $services_list_str = "";
                
                if ($services_res->num_rows > 0) {
                    include_once('jdf.php');
                    if (!function_exists('loginPanel')) include_once('class.php');
                    if (!function_exists('getClientTraffic')) include_once("api/sanaei.php");
                    
                    $services_list_str .= "\n\n➖➖➖ <b>سرویس‌ها</b> ➖➖➖";
                    while ($service = $services_res->fetch_assoc()) {
                        $stmt_server = $connect->prepare("SELECT * FROM servers WHERE name = ? LIMIT 1");
                        $stmt_server->bind_param("s", $service['svname']);
                        $stmt_server->execute();
                        $server_res = $stmt_server->get_result();
                        $server = $server_res->fetch_assoc();
                        $stmt_server->close();
                        
                        $used_gb = 0;
                        if ($server) {
                            // ... (traffic fetching logic remains the same)
                        }
                        
                        $total_gb = round($service['size'], 2);
                        $remaining_gb = max(0, $total_gb - $used_gb);
                        $volume_string = number_format($used_gb, 2) . " / " . number_format($total_gb, 2) . " گیگابایت (" . number_format($remaining_gb, 2) . " گیگ مانده)";
                        $days_remaining = max(0, floor(($service['expire'] - time()) / 86400));
                        $expire_date_jalali = jdate('Y/m/d', $service['expire']);
                        $is_active = (time() < $service['expire'] && $used_gb < $total_gb);

                        $services_list_str .= "\n\n🔹 <b>نام سرویس:</b> <code>{$service['display']}</code>";
                        $services_list_str .= "\n   - 📦 حجم: " . $volume_string;
                        $services_list_str .= "\n   - 📍 سرور: {$service['location']}";
                        $services_list_str .= "\n   - 🗓 انقضا: <code>{$expire_date_jalali}</code> ({$days_remaining} روز مانده)";
                        $services_list_str .= "\n   - 🚦 وضعیت: " . ($is_active ? "فعال" : "پایان یافته");
                    }
                } else {
                    $services_list_str = "\n\n- این کاربر هیچ سرویسی ندارد.";
                }
                $stmt_services->close();

                $user_display = $user_info['username'] ? "@" . $user_info['username'] : "ندارد";
                $message = "👤 <b>اطلاعات کاربر:</b>\n\n";
                $message .= "▫️ <b>ID:</b> <code>$user_id</code>\n";
                $message .= "▫️ <b>یوزرنیم:</b> $user_display\n";
                $message .= "▫️ <b>موجودی:</b> " . number_format($user_info['coin']) . " تومان";
                $message .= $services_list_str;

                sendMessage($chat_id, $message, $panel, 'html');
            } else {
                sendMessage($chat_id, "❌ کاربری با این مشخصات یافت نشد.", $panel);
            }
        }
        elseif ($text == "حذف سرویس کاربر") {
            $stmt = $connect->prepare("UPDATE user SET step = 'deletesv' WHERE id = ?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "🔢 آیدی عددی کاربر را وارد کنید:", $backp);
        }
        elseif ($user['step'] == "deletesv") {
            if (!is_numeric($text)) {
                sendMessage($chat_id, "❌ لطفاً فقط عدد وارد کنید.");
                exit();
            }
            $stmt = $connect->prepare("SELECT * FROM services WHERE owner = ?");
            $stmt->bind_param("s", $text);
            $stmt->execute();
            $get = $stmt->get_result();
            if ($get->num_rows == 0) {
                sendMessage($chat_id, "❌ هیچ سرویسی برای این کاربر یافت نشد.");
                exit();
            }
            $inlineKeyboard = [];
            $i = 0;
            while ($row = $get->fetch_assoc()) {
                $i++;
                $label = "$i-{$row['display']} - {$row['svname']} - {$row['size']} GB ❌";
                $inlineKeyboard[] = [['text' => $label, 'callback_data' => "addelete-{$row['id']}"]];
            }
            $stmt->close();
            $keyboard = json_encode(['inline_keyboard' => $inlineKeyboard]);
            sendMessage($chat_id, "🔘 روی سرویس مورد نظر کلیک کنید تا <b>از ربات و سرور حذف شود</b>:", $keyboard);
            
            $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
            $stmt_user->bind_param("s", $chat_id);
            $stmt_user->execute();
            $stmt_user->close();
        }
        elseif ($text == "مدیریت سرورها") {
            $result = $connect->query("SELECT * FROM servers ORDER BY id ASC");
            if ($result->num_rows > 0) {
                $keyboard = ['inline_keyboard' => []];
                while ($row = $result->fetch_assoc()) {
                    $status_icon = ($row['status'] == 'on') ? "🟢" : "🔴";
                    $hide_icon = $row['is_hidden'] ? '🙈' : '';
                    $test_icon = $row['is_test_server'] ? '🧪' : '';
                    $keyboard['inline_keyboard'][] = [['text' => "$status_icon {$row['display_name']} $hide_icon $test_icon", 'callback_data' => "manage_server|{$row['id']}"]];
                }
                $msg_text = "🎛️ <b>مدیریت سرورها</b>\n\nبرای مدیریت هر سرور (خاموش/روشن، حذف، مخفی کردن، انتخاب به عنوان سرور تست)، روی نام آن کلیک کنید.";
                sendMessage($chat_id, $msg_text, json_encode($keyboard), "html");
            } else {
                sendMessage($chat_id, "هیچ سروری برای مدیریت یافت نشد.", $panel);
            }
        }
        elseif ($text == "اضافه کردن سرور") {
            $stmt = $connect->prepare("UPDATE user SET step='add_server_name' WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "لطفاً یک نام داخلی (انگلیسی و بدون فاصله) برای سرور جدید وارد کنید (مثال: <code>de-hetzner</code>):", $backp, "html");
        }
        elseif ($user['step'] == 'add_server_name') {
            $step = "add_server_display_name|$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("s", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "حالا یک نام نمایشی (برای نمایش به کاربر) وارد کنید (مثال: آلمان ✨):", $backp);
        }
        elseif (preg_match('/^add_server_display_name\|(.+)/', $user['step'], $matches)) {
            $internal_name = $matches[1];
            $step = "add_server_type|$internal_name|$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("s", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "نوع پنل را وارد کنید (<code>marzban</code> یا <code>sanaei</code>):", $backp, "html");
        }
        elseif (preg_match('/^add_server_type\|(.+)\|(.+)/', $user['step'], $matches)) {
            $internal_name = $matches[1];
            $display_name = $matches[2];
            $step = "add_server_link|$internal_name|$display_name|$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("s", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "لینک پنل را وارد کنید (مثال: <code>https://panel.example.com</code>):", $backp, "html");
        }
        elseif (preg_match('/^add_server_link\|(.+)\|(.+)\|(.+)/', $user['step'], $matches)) {
            $internal_name = $matches[1];
            $display_name = $matches[2];
            $type = $matches[3];
            $step = "add_server_user|$internal_name|$display_name|$type|$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("s", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "نام کاربری پنل را وارد کنید (اگر ندارد <code>none</code> بفرستید):", $backp, "html");
        }
        elseif (preg_match('/^add_server_user\|(.+)\|(.+)\|(.+)\|(.+)/', $user['step'], $matches)) {
            $internal_name = $matches[1];
            $display_name = $matches[2];
            $type = $matches[3];
            $link = $matches[4];
            $step = "add_server_pass|$internal_name|$display_name|$type|$link|$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("s", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "رمز عبور پنل را وارد کنید (اگر ندارد <code>none</code> بفرستید):", $backp, "html");
        }
        elseif (preg_match('/^add_server_pass\|(.+)\|(.+)\|(.+)\|(.+)\|(.+)/', $user['step'], $matches)) {
            $internal_name = $matches[1];
            $display_name = $matches[2];
            $type = $matches[3];
            $link = $matches[4];
            $user_panel = $matches[5];
            $pass_panel = $text;

            $stmt = $connect->prepare("INSERT INTO servers (name, display_name, type, link, username, password, status) VALUES (?, ?, ?, ?, ?, ?, 'on')");
            $stmt->bind_param("ssssss", $internal_name, $display_name, $type, $link, $user_panel, $pass_panel);
            if ($stmt->execute()) {
                sendMessage($chat_id, "✅ سرور با موفقیت اضافه شد.", $panel);
            } else {
                sendMessage($chat_id, "❌ خطا در افزودن سرور: " . $stmt->error, $panel);
            }
            $stmt->close();

            $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
            $stmt_user->bind_param("s", $chat_id);
            $stmt_user->execute();
            $stmt_user->close();
        }
        elseif ($text == "مدیریت تعرفه") {
            $plans_res = $connect->query("SELECT * FROM plans ORDER BY id ASC");
            if ($plans_res->num_rows > 0) {
                $keyboard = ['inline_keyboard' => []];
                while($plan = $plans_res->fetch_assoc()){
                    $keyboard['inline_keyboard'][] = [['text' => "{$plan['title']} ({$plan['volume']}GB - {$plan['duration']} روز) 🗑", 'callback_data' => "addelete-plan-{$plan['id']}"]];
                }
                sendMessage($chat_id, "برای حذف هر تعرفه، روی آن کلیک کنید:", json_encode($keyboard));
            } else {
                sendMessage($chat_id, "هیچ تعرفه‌ای برای مدیریت یافت نشد.", $panel);
            }
        }
        elseif ($text == "افزودن تعرفه") {
            $stmt = $connect->prepare("UPDATE user SET step='add_plan_title' WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "لطفاً عنوان تعرفه را وارد کنید (مثال: پلن یک ماهه):", $backp);
        }
        elseif ($user['step'] == 'add_plan_title') {
            $step = "add_plan_volume|$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("s", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "حجم تعرفه را به گیگابایت (فقط عدد) وارد کنید:", $backp);
        }
        elseif (preg_match('/^add_plan_volume\|(.+)/', $user['step'], $matches)) {
            if (!is_numeric($text)) {
                sendMessage($chat_id, "❌ لطفاً فقط عدد وارد کنید.");
                exit();
            }
            $title = $matches[1];
            $step = "add_plan_duration|$title|$text";
            $stmt = $connect->prepare("UPDATE user SET step=? WHERE id=?");
            $stmt->bind_param("s", $step, $chat_id);
            $stmt->execute();
            $stmt->close();
            sendMessage($chat_id, "مدت زمان تعرفه را به روز (فقط عدد) وارد کنید:", $backp);
        }
        elseif (preg_match('/^add_plan_duration\|(.+)\|(\d+)/', $user['step'], $matches)) {
            if (!is_numeric($text)) {
                sendMessage($chat_id, "❌ لطفاً فقط عدد وارد کنید.");
                exit();
            }
            $title = $matches[1];
            $volume = (int)$matches[2];
            $duration = (int)$text;

            $stmt = $connect->prepare("INSERT INTO plans (title, volume, duration) VALUES (?, ?, ?)");
            $stmt->bind_param("sii", $title, $volume, $duration);
            if ($stmt->execute()) {
                sendMessage($chat_id, "✅ تعرفه با موفقیت اضافه شد.", $panel);
            } else {
                sendMessage($chat_id, "❌ خطا در افزودن تعرفه: " . $stmt->error, $panel);
            }
            $stmt->close();

            $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
            $stmt_user->bind_param("s", $chat_id);
            $stmt_user->execute();
            $stmt_user->close();
        }
        elseif ($text == '⚙️ مدیریت قیمت‌ها') {
            $servers_res = $connect->query("SELECT * FROM servers WHERE status = 'on' AND is_hidden = 0 ORDER BY id ASC");
            if ($servers_res->num_rows > 0) {
                $keyboard = ['inline_keyboard' => []];
                while($server = $servers_res->fetch_assoc()) {
                    $keyboard['inline_keyboard'][] = [['text' => $server['display_name'], 'callback_data' => "set_price_for_server|{$server['id']}"]];
                }
                sendMessage($chat_id, "لطفاً سروری که می‌خواهید قیمت‌های آن را تنظیم کنید، انتخاب نمایید:", json_encode($keyboard));
            } else {
                sendMessage($chat_id, "هیچ سرور فعالی برای قیمت‌گذاری یافت نشد.", $panel);
            }
        }
        elseif ($text == 'خاموش/روشن اد اجباری') {
            $current_status = get_setting($connect, 'join_status');
            $new_status = ($current_status == 'on') ? 'off' : 'on';
            $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('join_status', ?) ON DUPLICATE KEY UPDATE `value` = ?");
            $stmt->bind_param("ss", $new_status, $new_status);
            $stmt->execute();
            $stmt->close();
            $status_text = ($new_status == 'on') ? 'روشن' : 'خاموش';
            sendMessage($chat_id, "✅ عضویت اجباری با موفقیت $status_text شد.", $panel);
        }
        elseif ($text == 'تغییر شماره کارت') {
            $stmt = $connect->prepare("UPDATE user SET step='set_card_number' WHERE id=?");
            $stmt->bind_param("s", $chat_id);
            $stmt->execute();
            $stmt->close();
            $current_card = get_setting($connect, 'card_info');
            sendMessage($chat_id, "لطفاً شماره کارت جدید را به همراه نام صاحب حساب وارد کنید.\n\nشماره کارت فعلی:\n<code>$current_card</code>", $backp, "html");
        }
        elseif ($user['step'] == 'set_card_number') {
            $stmt = $connect->prepare("INSERT INTO settings (`key`, `value`) VALUES ('card_info', ?) ON DUPLICATE KEY UPDATE `value` = ?");
            $stmt->bind_param("ss", $text, $text);
            $stmt->execute();
            $stmt->close();
            
            $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
            $stmt_user->bind_param("s", $chat_id);
            $stmt_user->execute();
            $stmt_user->close();
            
            sendMessage($chat_id, "✅ شماره کارت با موفقیت بروزرسانی شد.", $panel);
        }
        else {
             sendMessage($chat_id, "دستور نامشخص است.", $panel);
        }
        exit();
    }
    
    if($user['step'] == 'waiting_kyc_photo' && isset($photoid)){
        $kyc_channel_id = get_setting($connect, 'kyc_channel_id');
        $admin_id_for_kyc = get_setting($connect, 'admin_id');

        if(empty($kyc_channel_id) || empty($admin_id_for_kyc)){
            sendMessage($chat_id, "❌ خطای سیستمی: کانال یا ادمین احراز هویت در تنظیمات ربات تعریف نشده است. لطفاً به مدیر اطلاع دهید.", $start);
            exit();
        }

        $user_info_caption = "👤 کاربر با آیدی عددی <code>$chat_id</code> درخواست احراز هویت دارد.";

        bot('sendPhoto', [
            'chat_id' => $kyc_channel_id,
            'photo' => $photoid,
            'caption' => $user_info_caption,
            'parse_mode' => 'html',
            'reply_markup' => json_encode(['inline_keyboard' => [
                [['text' => "✅ تایید هویت", 'callback_data' => "kyc_approve|$chat_id"]],
                [['text' => "❌ رد هویت", 'callback_data' => "kyc_reject|$chat_id"]]
            ]])
        ]);

        sendMessage($chat_id, "✅ مدارک شما جهت بررسی برای مدیریت ارسال شد. نتیجه به زودی به شما اطلاع داده خواهد شد.", $start);
        $stmt = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
        $stmt->bind_param("s", $chat_id);
        $stmt->execute();
        $stmt->close();
        exit();
    }
    if($text =="🛒 خرید سرویس"){
        $stmt = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
        $stmt->bind_param("s", $chat_id);
        $stmt->execute();
        $stmt->close();
        $keyboard = ['inline_keyboard' => []];
        $result = $connect->query("SELECT id, display_name FROM servers WHERE status='on' AND is_hidden = 0 AND is_test_server = 0 ORDER BY id ASC");
        while ($row = $result->fetch_assoc()) $keyboard['inline_keyboard'][] = [['text' => "✨ {$row['display_name']} ✨", 'callback_data' => "selectserver|{$row['id']}"]];
        if (empty($keyboard['inline_keyboard'])) sendMessage($chat_id, getMessage($connect, 'error_no_active_server_for_buy'));
        else {
            $response = sendmessage($chat_id, getMessage($connect, 'ask_location'), json_encode($keyboard));
            if ($msgid = $response['result']['message_id'] ?? null) {
                $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                $stmt_msg->bind_param("is", $msgid, $chat_id);
                $stmt_msg->execute();
                $stmt_msg->close();
            }
        }
    }
    elseif($text=="🗄 سرویس ها"){
        $stmt = $connect->prepare("SELECT id, display, size FROM services WHERE owner = ?");
        $stmt->bind_param("s", $from_id);
        $stmt->execute();
        $get_services = $stmt->get_result();
        if($get_services->num_rows == 0){ sendMessage($chat_id, getMessage($connect, 'error_no_service_found_for_user')); exit(); }
        $keys = []; $i = 0;
        while ($service = $get_services->fetch_assoc()) {
            $i++; $keys[] = [['text' => "$i-{$service['display']} - {$service['size']} GB", 'callback_data' => "service-{$service['id']}"]];
        }
        $stmt->close();
        $response = sendmessage($chat_id, getMessage($connect, 'service_list_prompt'), json_encode(['inline_keyboard' => $keys]));
        if ($msgid = $response['result']['message_id'] ?? null) {
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $msgid, $chat_id);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
    }
    elseif($text=="♻️ تمدید سرویس"){
        $stmt = $connect->prepare("SELECT id, display, size FROM services WHERE owner = ?");
        $stmt->bind_param("s", $from_id);
        $stmt->execute();
        $get_services = $stmt->get_result();
        if($get_services->num_rows == 0){ sendMessage($chat_id, getMessage($connect, 'error_no_service_found_for_user')); exit(); }
        $keys = []; $i = 0;
        while ($service = $get_services->fetch_assoc()) {
            $i++; $keys[] = [['text' => "$i-{$service['display']} - {$service['size']} GB", 'callback_data' => "ex-{$service['id']}"]];
        }
        $stmt->close();
        $response = sendmessage($chat_id, getMessage($connect, 'extend_service_prompt'), json_encode(['inline_keyboard' => $keys]));
        if ($msgid = $response['result']['message_id'] ?? null) {
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $msgid, $chat_id);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
    } 
    elseif ($text == "🎁 دعوت دوستان") {
        $settings = function_exists('referral_get_settings') ? referral_get_settings($connect) : ['status' => 'off'];
        if (($settings['status'] ?? 'off') !== 'on') {
            $disabled = getMessage($connect, 'referral_disabled_message');
            sendMessage($chat_id, $disabled, $start, 'html');
        } else {
            $code = function_exists('referral_ensure_user_code') ? referral_ensure_user_code($connect, (string)$chat_id) : '';
            $link = ($code && function_exists('referral_build_link')) ? referral_build_link($connect, $code) : null;
            $stats = function_exists('referral_get_user_stats') ? referral_get_user_stats($connect, (string)$chat_id) : ['invited' => 0, 'successful' => 0, 'pending' => 0];
            $reward_type = $settings['reward_type'] ?? 'fixed';
            $value = (float)($settings['reward_value'] ?? 0);
            if ($reward_type === 'percent') {
                $reward_value_text = number_format(max(1, min(100, $value))) . '%';
                $reward_type_label = 'درصد';
            } else {
                $reward_value_text = referral_format_currency(max(0, $value));
                $reward_type_label = 'تومان';
            }
            $safeLink = $link ? escape_html($link) : '—';
            $safeCode = $code ? escape_html($code) : '—';
            $message = getMessage($connect, 'referral_share_message', [
                'referral_link' => $safeLink,
                'referral_code' => $safeCode,
                'reward_value' => $reward_value_text,
                'reward_type_label' => $reward_type_label,
                'referral_total_invited' => number_format($stats['invited'] ?? 0),
                'referral_success_count' => number_format($stats['successful'] ?? 0),
                'referral_pending_count' => number_format($stats['pending'] ?? 0),
            ]);
            sendMessage($chat_id, $message, $start, 'html');
        }
        exit();
    }
    elseif ($text == "🪪اکانت تست") {
        handleTestAccountRequest($connect, $chat_id, $user, $admin_id);
    } 
    elseif (preg_match('/^change-(\d+)/', $user['step'], $matches)) {
        $service_id = $matches[1];
        deleteMessage($chat_id, $user['msgid']);
        $message_text = getMessage($connect, 'service_name_changed_success', ['new_name' => $text]);
        sendmessage($chat_id, $message_text, $start, "html");
        $stmt = $connect->prepare("UPDATE services SET display=?, last_name_change_at=? WHERE id=?");
        $time = time();
        $stmt->bind_param("sis", $text, $time, $service_id);
        $stmt->execute();
        $stmt->close();
        
        $stmt_user = $connect->prepare("UPDATE user SET step='none' WHERE id=?");
        $stmt_user->bind_param("s", $chat_id);
        $stmt_user->execute();
        $stmt_user->close();
    }
    elseif ( ($user['step'] == 'waiting_payment_card' || $user['step'] == 'waiting_payment_crypto') && isset($photoid) ) {
        $order_code = $user['tmp'];
        $stmt = $connect->prepare("SELECT * FROM orders WHERE order_code = ? LIMIT 1");
        $stmt->bind_param("s", $order_code);
        $stmt->execute();
        $order_res = $stmt->get_result();
        
        if ($order_res && $order_res->num_rows > 0) {
            $order = $order_res->fetch_assoc();
            $type = ($order['type'] == 'exten') ? 'تمدید' : 'خرید';
            $user_info = $from_id['first_name'] . " " . ($from_id['last_name'] ?? '');
            $username = isset($from_id['username']) ? " (@{$from_id['username']})" : "";
            $admin_caption = "";
            if($user['step'] == 'waiting_payment_card'){
                $admin_caption = "🧾 <b>فیش واریزی (کارت به کارت)</b>\n\n" . "🔗 <b>نوع:</b> $type\n" . "👤 <b>کاربر:</b> $chat_id$username\n" . "📦 <b>حجم:</b> {$order['size']}G\n" . "💸 <b>مبلغ:</b> ".number_format($order['amount'])." تومان\n" . "📍 <b>لوکیشن:</b> {$order['location']}\n" . "🔑 <b>کد سفارش:</b> <code>$order_code</code>\n";
                if ((int)($order['discount_amount'] ?? 0) > 0) {
                    $admin_caption .= "🎟️ <b>تخفیف:</b> ".number_format($order['discount_amount'])." تومان (کد: <code>".($order['discount_code'] ?? '')."</code>)\n";
                }
                $admin_caption .= "\n";
            } else {
                $usdt_price = get_usdt_price();
                $amount_toman = $order['amount'];
                $amount_text = number_format($amount_toman) . " تومان";
                if($usdt_price){
                    // Display USDT amount with a 5% markup for clarity
                    $amount_usdt = round(($amount_toman / $usdt_price) * 1.05, 2);
                    $amount_text .= " / <code>$amount_usdt</code> تتر";
                }
                $admin_caption = "🧾 <b>رسید پرداخت (رمز ارز)</b>\n\n" . "🔗 <b>نوع:</b> $type\n" . "👤 <b>کاربر:</b> <code>$chat_id</code>$username\n" . "📦 <b>حجم:</b> {$order['size']}G\n" . "💸 <b>مبلغ:</b> $amount_text\n" . "📍 <b>لوکیشن:</b> {$order['location']}\n" . "🔑 <b>کد سفارش:</b> <code>$order_code</code>\n\n";
                if(!empty($caption)){
                    $admin_caption .= "🔗 <b>لینک/هش تراکنش:</b>\n<code>$caption</code>\n\n";
                } else {
                    $admin_caption .= "⚠️ <b>کاربر لینک تراکنش ارسال نکرده است.</b>\n\n";
                }
            }
            $admin_caption .= "تایید شود؟";
            bot('sendPhoto', [ 'chat_id' => $admin_id, 'photo' => $photoid, 'caption' => $admin_caption, 'parse_mode' => 'html', 'reply_markup' => json_encode(['inline_keyboard' => [[['text' => "✅", 'callback_data' => "approve|$order_code"]], [['text' => "❌", 'callback_data' => "reject|$order_code"]]]]) ]);
            $ack = getMessage($connect, 'payment_receipt_received');
            if (strpos($ack, 'خطای پیام') === 0) { $ack = '✅ رسید شما دریافت و برای بررسی به مدیریت ارسال شد.'; }
            sendmessage($chatid, $ack, $start);
            $stmt_user = $connect->prepare("UPDATE user SET step='none', tmp=NULL WHERE id=?");
            $stmt_user->bind_param("s", $chatid);
            $stmt_user->execute();
            $stmt_user->close();
        } else {
            sendmessage($chatid, "خطایی در پردازش رسید شما رخ داد. لطفاً دوباره مراحل خرید را طی کنید.", $start);
        }
        $stmt->close();
    }
    elseif ($text=="👤 حساب کاربری") {
        $stmt = $connect->prepare("SELECT COUNT(*) as count FROM services WHERE owner = ?");
        $stmt->bind_param("s", $chat_id);
        $stmt->execute();
        $res = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        sendmessage($chat_id, getMessage($connect, 'user_account_info', ['user_id' => $chat_id, 'service_count' => $res['count']]), null, "html");
    }
    elseif ($text == "📋 تعرفه سرویس ها") {
        $keyboard = ['inline_keyboard' => []];
        $result = $connect->query("SELECT id, display_name FROM servers WHERE status='on' AND is_hidden = 0 AND is_test_server = 0 ORDER BY id ASC");
        if ($result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) $keyboard['inline_keyboard'][] = [['text' => "✨ {$row['display_name']} ✨", 'callback_data' => "tariffs_for_server|{$row['id']}"]];
            $response = sendMessage($chat_id, "⭐️ لطفاً برای مشاهده تعرفه‌ها، لوکیشن مورد نظر خود را انتخاب کنید:", json_encode($keyboard));
            if ($msgid = $response['result']['message_id'] ?? null) {
                $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
                $stmt_msg->bind_param("is", $msgid, $chat_id);
                $stmt_msg->execute();
                $stmt_msg->close();
            }
        } else {
            sendMessage($chat_id, getMessage($connect, 'error_no_active_server_for_tariff'));
        }
    }
    elseif ($text=="☎️ پشتیبانی") {
        sendmessage($chat_id, getMessage($connect, 'contact_support', ['admin_contact' => getMessage($connect, 'support_contact_info')]));
    }
    exit();
}

// =========================================================================================================
// --- DEPRECATED & COMMENTED OUT CODE ---
// =========================================================================================================
/*
if(preg_match('/^confirm/',$data)){
    // This block was a previous implementation for purchase using wallet balance (coin)
    // It has been replaced by the payment gateway flow (orders table)
}
*/

/*
elseif($text=="🎁 کد تخفیف"){
    $connect->query("UPDATE user SET step='gift' WHERE id=?");
    sendmessage($chat_id,"کد کارت هدیه را وارد کنید",$back,'html');
}
elseif($user['step']=="gift"){
    if(is_file("gift/$text")){
        if(!is_file("gifts/$chat_id/$text")){
            mkdir("gifts");
            mkdir("gifts/$chat_id");
            touch("gifts/$chat_id/$text");
            $amount=file_get_contents("gift/$text");
            $connect->query("UPDATE user SET coin=coin+'$amount',step='none' WHERE id=?");
            sendmessage($chat_id,"مبلغ $amount تومان به حساب شما افزوده شد!",$start);
        }
    } else {
    // Wallet menu via text button (tolerant to emoji variations)
    if (isset($text) && (strpos($text, 'کیف پول') !== false)) {
        $stmtb = $connect->prepare("SELECT coin FROM user WHERE id = ? LIMIT 1");
        $stmtb->bind_param("s", $chat_id);
        $stmtb->execute();
        $rowb = $stmtb->get_result()->fetch_assoc();
        $stmtb->close();
        $balance_text = number_format((int)($rowb['coin'] ?? 0));
        $msg = "👛 <b>کیف پول شما</b>\n\n💰 موجودی: <b>{$balance_text}</b> تومان";
        $kb = json_encode(['inline_keyboard' => [
            [ ['text' => '➕ شارژ کارت به کارت', 'callback_data' => 'wallet_charge_card'], ['text' => '➕ شارژ رمز ارز', 'callback_data' => 'wallet_charge_crypto'] ],
            [ ['text' => '🔙 بازگشت', 'callback_data' => 'buy'] ]
        ]]);
        $sent = sendMessage($chat_id, $msg, $kb, 'html');
        $msgid_wallet = $sent['result']['message_id'] ?? null;
        if ($msgid_wallet) {
            $stmt_msg = $connect->prepare("UPDATE user SET msgid=? WHERE id=?");
            $stmt_msg->bind_param("is", $msgid_wallet, $chat_id);
            $stmt_msg->execute();
            $stmt_msg->close();
        }
        exit();
    }

    // Await wallet card amount
    if (($user['step'] ?? '') === 'await_wallet_card_amount' && isset($text)) {
        $amount = (int)preg_replace('/[^0-9]/', '', (string)$text);
        if ($amount <= 0) { sendMessage($chat_id, '❌ لطفاً فقط رقم معتبر وارد کنید.'); exit(); }
        $connect->query("CREATE TABLE IF NOT EXISTS wallet_topups (id INT AUTO_INCREMENT PRIMARY KEY, code VARCHAR(32) UNIQUE, chat_id BIGINT, amount INT, method ENUM('card','crypto') NOT NULL, status ENUM('pending','approved','rejected') DEFAULT 'pending', created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP)");
        $code = strtoupper(substr(md5(uniqid('', true)), 0, 8));
        $stmt_i = $connect->prepare("INSERT INTO wallet_topups (code, chat_id, amount, method, status) VALUES (?, ?, ?, 'card', 'pending')");
        $stmt_i->bind_param('sii', $code, $chat_id, $amount);
        $stmt_i->execute();
        $stmt_i->close();
        $stmt_user = $connect->prepare("UPDATE user SET step='waiting_wallet_card', tmp=? WHERE id=?");
        $stmt_user->bind_param('ss', $code, $chat_id);
        $stmt_user->execute();
        $stmt_user->close();
        $card_info = get_setting($connect, 'card_info');
        $back_key = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => 'wallet']]]]);
        $msg = "💳 <b>کارت به کارت</b>\n\nمبلغ شارژ: <b>" . number_format($amount) . "</b> تومان\nکد پیگیری: <code>{$code}</code>\n\nلطفاً به کارت زیر واریز و سپس <b>اسکرین‌شات رسید</b> را همین‌جا ارسال کنید.\n\n<code>{$card_info}</code>";
        sendMessage($chat_id, $msg, $back_key, 'html');
        exit();
    }

    // Await wallet crypto amount
    if (($user['step'] ?? '') === 'await_wallet_crypto_amount' && isset($text)) {
        $amount = (int)preg_replace('/[^0-9]/', '', (string)$text);
        if ($amount <= 0) { sendMessage($chat_id, '❌ لطفاً فقط رقم معتبر وارد کنید.'); exit(); }
        $wallets = get_setting($connect, 'crypto_wallets');
        if (empty($wallets)) { sendMessage($chat_id, '⛔️ هیچ آدرس کیف پولی تنظیم نشده است.'); exit(); }
        $connect->query("CREATE TABLE IF NOT EXISTS wallet_topups (id INT AUTO_INCREMENT PRIMARY KEY, code VARCHAR(32) UNIQUE, chat_id BIGINT, amount INT, method ENUM('card','crypto') NOT NULL, status ENUM('pending','approved','rejected') DEFAULT 'pending', created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP)");
        $code = strtoupper(substr(md5(uniqid('', true)), 0, 8));
        $stmt_i = $connect->prepare("INSERT INTO wallet_topups (code, chat_id, amount, method, status) VALUES (?, ?, ?, 'crypto', 'pending')");
        $stmt_i->bind_param('sii', $code, $chat_id, $amount);
        $stmt_i->execute();
        $stmt_i->close();
        $stmt_user = $connect->prepare("UPDATE user SET step='waiting_wallet_crypto', tmp=? WHERE id=?");
        $stmt_user->bind_param('ss', $code, $chat_id);
        $stmt_user->execute();
        $stmt_user->close();
        $back_key = json_encode(['inline_keyboard' => [[['text' => '🔙 بازگشت', 'callback_data' => 'wallet']]]]);
        $msg = "💎 <b>شارژ با رمز ارز</b>\n\nمبلغ شارژ: <b>" . number_format($amount) . "</b> تومان\nکد پیگیری: <code>{$code}</code>\n\nمبلغ معادل را به یکی از آدرس‌های زیر واریز کنید و سپس <b>اسکرین‌شات رسید</b> را همراه با <b>TxID</b> ارسال نمایید:\n\n<code>\n{$wallets}\n</code>";
        sendMessage($chat_id, $msg, $back_key, 'html');
        exit();
    }
        sendmessage($chat_id,"کارت هدیه وجود ندارد");
    }
}
*/

/*
elseif($text=="❓ سوالات متداول"){
    sendmessage($chat_id,"⁉️به بخش سوالات متداول خوش آمدید⁉️

 ❗️ایپی اکانت های شما ثابت است؟ 
+بله 

❗️اشتراک های شما چند کاربر هستند؟
+نامحدود کاربر

  ❗️منظور از اپدیت سابسکریپشن چیست؟
+نحوه این کار در تمام ویدیوهای آموزشی بیان شده ؛ شما با این کار میتوانید در صورت اعلام از سمت ما، کانفیگ های جدید اضافه کنید. 


❗️نحوه مشاهده حجم و تاریخ اتمام: 
در دو قسمت قابل مشاهده است 
هم از طریق بات و هم از طریق یکی از کانفیگ های لیست.");
    $key=json_encode(['inline_keyboard'=>[
        [['text'=>'👇Ios👇','callback_data'=>'none'],['text'=>"👇Android👇",'callback_data'=>"none"],['text'=>'👇Windows👇','callback_data'=>'none']],
        [['text'=>"Streisand",'url'=>"https://t.me/boltvpn2/3"],['text'=>"v2rayNg",'url'=>"https://t.me/boltvpn2/6"],['text'=>"nekoray",'url'=>"https://t.me/boltvpn2/7"]],
        [['text'=>"V2Box",'url'=>"https://t.me/boltvpn2/4"],['text'=>"-",'callback_data'=>"none"],['text'=>"v2rayN",'url'=>"https://t.me/boltvpn2/8"]],
        [['text'=>"fair vpn",'url'=>"https://t.me/boltvpn2/5"],['text'=>"-",'callback_data'=>"none"],['text'=>"-",'callback_data'=>"none"]],
    ]]);
    sendmessage($chat_id,"آموزش برنامه های مورد نیاز:",$key);
}
*/

?>
