<?php
// Utility functions for managing per-server custom domains

/**
 * Ensure the servers table has a custom_domain column for overriding subscription domains.
 */
function ensure_server_domain_column(mysqli $connect): void {
    $res = $connect->query("SHOW COLUMNS FROM servers LIKE 'custom_domain'");
    if ($res && $res->num_rows === 0) {
        $connect->query("ALTER TABLE servers ADD COLUMN custom_domain VARCHAR(255) NULL");
    }
}

/**
 * Replace the host of a URL with the server's custom_domain if one is set.
 *
 * @param string $url     Original URL
 * @param array  $server  Server array containing optional 'custom_domain'
 * @return string Modified URL
 */
function apply_custom_domain(?string $url, array $server): string {
    if (!$url) return '';
    $custom = $server['custom_domain'] ?? '';
    if (empty($custom)) return $url;
    $parsed = parse_url($url);
    if (!isset($parsed['host'])) return $url;
    return str_replace($parsed['host'], $custom, $url);
}

/**
 * Apply custom domain based on the URL's host by looking up the server entry.
 *
 * @param mysqli $connect Database connection
 * @param string $url     Original URL
 * @return string Modified URL
 */
function apply_custom_domain_by_link(mysqli $connect, ?string $url): string {
    if (!$url) return '';
    $parsed = parse_url($url);
    if (!isset($parsed['host'])) return $url;
    $host = $parsed['host'];
    $stmt = $connect->prepare("SELECT custom_domain FROM servers WHERE link LIKE ? LIMIT 1");
    $like = "%$host%";
    $stmt->bind_param('s', $like);
    $stmt->execute();
    $res = $stmt->get_result();
    $server = $res ? $res->fetch_assoc() : null;
    $stmt->close();
    if ($server && !empty($server['custom_domain'])) {
        return str_replace($host, $server['custom_domain'], $url);
    }
    return $url;
}
